from __future__ import annotations

from abc import ABC, abstractmethod
from enum import Enum
from types import NoneType
from typing import TYPE_CHECKING, Any

from plain.exceptions import ValidationError
from plain.models.exceptions import FieldError
from plain.models.expressions import Exists, ExpressionList, F, OrderBy
from plain.models.indexes import IndexExpression
from plain.models.lookups import Exact
from plain.models.query_utils import Q
from plain.models.sql.query import Query

if TYPE_CHECKING:
    from plain.models.backends.base.schema import BaseDatabaseSchemaEditor
    from plain.models.backends.ddl_references import Statement
    from plain.models.base import Model
    from plain.models.expressions import Combinable

__all__ = ["BaseConstraint", "CheckConstraint", "Deferrable", "UniqueConstraint"]


class BaseConstraint(ABC):
    default_violation_error_message = 'Constraint "%(name)s" is violated.'
    violation_error_code: str | None = None
    violation_error_message: str | None = None

    def __init__(
        self,
        *,
        name: str,
        violation_error_code: str | None = None,
        violation_error_message: str | None = None,
    ) -> None:
        self.name = name
        if violation_error_code is not None:
            self.violation_error_code = violation_error_code
        if violation_error_message is not None:
            self.violation_error_message = violation_error_message
        else:
            self.violation_error_message = self.default_violation_error_message

    @property
    def contains_expressions(self) -> bool:
        return False

    @abstractmethod
    def constraint_sql(
        self, model: type[Model], schema_editor: BaseDatabaseSchemaEditor
    ) -> str | None: ...

    @abstractmethod
    def create_sql(
        self, model: type[Model], schema_editor: BaseDatabaseSchemaEditor
    ) -> str | Statement | None: ...

    @abstractmethod
    def remove_sql(
        self, model: type[Model], schema_editor: BaseDatabaseSchemaEditor
    ) -> str | Statement | None: ...

    @abstractmethod
    def validate(
        self, model: type[Model], instance: Model, exclude: set[str] | None = None
    ) -> None: ...

    def get_violation_error_message(self) -> str:
        return self.violation_error_message % {"name": self.name}  # type: ignore[operator]

    def deconstruct(self) -> tuple[str, tuple[Any, ...], dict[str, Any]]:
        path = f"{self.__class__.__module__}.{self.__class__.__name__}"
        path = path.replace("plain.models.constraints", "plain.models")
        kwargs: dict[str, Any] = {"name": self.name}
        if (
            self.violation_error_message is not None
            and self.violation_error_message != self.default_violation_error_message
        ):
            kwargs["violation_error_message"] = self.violation_error_message
        if self.violation_error_code is not None:
            kwargs["violation_error_code"] = self.violation_error_code
        return (path, (), kwargs)

    def clone(self) -> BaseConstraint:
        _, args, kwargs = self.deconstruct()
        return self.__class__(*args, **kwargs)


class CheckConstraint(BaseConstraint):
    def __init__(
        self,
        *,
        check: Q,
        name: str,
        violation_error_code: str | None = None,
        violation_error_message: str | None = None,
    ) -> None:
        self.check = check
        if not getattr(check, "conditional", False):
            raise TypeError(
                "CheckConstraint.check must be a Q instance or boolean expression."
            )
        super().__init__(
            name=name,
            violation_error_code=violation_error_code,
            violation_error_message=violation_error_message,
        )

    def _get_check_sql(
        self, model: type[Model], schema_editor: BaseDatabaseSchemaEditor
    ) -> str:
        query = Query(model=model, alias_cols=False)
        where = query.build_where(self.check)
        compiler = query.get_compiler()
        sql, params = where.as_sql(compiler, schema_editor.connection)
        return sql % tuple(schema_editor.quote_value(p) for p in params)

    def constraint_sql(
        self, model: type[Model], schema_editor: BaseDatabaseSchemaEditor
    ) -> str:
        check = self._get_check_sql(model, schema_editor)
        return schema_editor._check_sql(self.name, check)

    def create_sql(
        self, model: type[Model], schema_editor: BaseDatabaseSchemaEditor
    ) -> Statement | None:
        check = self._get_check_sql(model, schema_editor)
        return schema_editor._create_check_sql(model, self.name, check)

    def remove_sql(
        self, model: type[Model], schema_editor: BaseDatabaseSchemaEditor
    ) -> Statement | None:
        return schema_editor._delete_check_sql(model, self.name)

    def validate(
        self, model: type[Model], instance: Model, exclude: set[str] | None = None
    ) -> None:
        against = instance._get_field_value_map(meta=model._model_meta, exclude=exclude)
        try:
            if not Q(self.check).check(against):
                raise ValidationError(
                    self.get_violation_error_message(), code=self.violation_error_code
                )
        except FieldError:
            pass

    def __repr__(self) -> str:
        return "<{}: check={} name={}{}{}>".format(
            self.__class__.__qualname__,
            self.check,
            repr(self.name),
            (
                ""
                if self.violation_error_code is None
                else f" violation_error_code={self.violation_error_code!r}"
            ),
            (
                ""
                if self.violation_error_message is None
                or self.violation_error_message == self.default_violation_error_message
                else f" violation_error_message={self.violation_error_message!r}"
            ),
        )

    def __eq__(self, other: object) -> bool:
        if isinstance(other, CheckConstraint):
            return (
                self.name == other.name
                and self.check == other.check
                and self.violation_error_code == other.violation_error_code
                and self.violation_error_message == other.violation_error_message
            )
        return super().__eq__(other)

    def deconstruct(self) -> tuple[str, tuple[Any, ...], dict[str, Any]]:
        path, args, kwargs = super().deconstruct()
        kwargs["check"] = self.check
        return path, args, kwargs


class Deferrable(Enum):
    DEFERRED = "deferred"
    IMMEDIATE = "immediate"

    # A similar format was proposed for Python 3.10.
    def __repr__(self) -> str:
        return f"{self.__class__.__qualname__}.{self._name_}"


class UniqueConstraint(BaseConstraint):
    def __init__(
        self,
        *expressions: str | Combinable,
        fields: tuple[str, ...] | list[str] = (),
        name: str | None = None,
        condition: Q | None = None,
        deferrable: Deferrable | None = None,
        include: tuple[str, ...] | list[str] | None = None,
        opclasses: tuple[str, ...] | list[str] = (),
        violation_error_code: str | None = None,
        violation_error_message: str | None = None,
    ) -> None:
        if not name:
            raise ValueError("A unique constraint must be named.")
        if not expressions and not fields:
            raise ValueError(
                "At least one field or expression is required to define a "
                "unique constraint."
            )
        if expressions and fields:
            raise ValueError(
                "UniqueConstraint.fields and expressions are mutually exclusive."
            )
        if not isinstance(condition, NoneType | Q):
            raise ValueError("UniqueConstraint.condition must be a Q instance.")
        if condition and deferrable:
            raise ValueError("UniqueConstraint with conditions cannot be deferred.")
        if include and deferrable:
            raise ValueError("UniqueConstraint with include fields cannot be deferred.")
        if opclasses and deferrable:
            raise ValueError("UniqueConstraint with opclasses cannot be deferred.")
        if expressions and deferrable:
            raise ValueError("UniqueConstraint with expressions cannot be deferred.")
        if expressions and opclasses:
            raise ValueError(
                "UniqueConstraint.opclasses cannot be used with expressions. "
                "Use a custom OpClass() instead."
            )
        if not isinstance(deferrable, NoneType | Deferrable):
            raise ValueError(
                "UniqueConstraint.deferrable must be a Deferrable instance."
            )
        if not isinstance(include, NoneType | list | tuple):
            raise ValueError("UniqueConstraint.include must be a list or tuple.")
        if not isinstance(opclasses, list | tuple):
            raise ValueError("UniqueConstraint.opclasses must be a list or tuple.")
        if opclasses and len(fields) != len(opclasses):
            raise ValueError(
                "UniqueConstraint.fields and UniqueConstraint.opclasses must "
                "have the same number of elements."
            )
        self.fields = tuple(fields)
        self.condition = condition
        self.deferrable = deferrable
        self.include = tuple(include) if include else ()
        self.opclasses = opclasses
        self.expressions = tuple(
            F(expression) if isinstance(expression, str) else expression
            for expression in expressions
        )
        super().__init__(
            name=name,  # type: ignore[arg-type]
            violation_error_code=violation_error_code,
            violation_error_message=violation_error_message,
        )

    @property
    def contains_expressions(self) -> bool:
        return bool(self.expressions)

    def _get_condition_sql(
        self, model: type[Model], schema_editor: BaseDatabaseSchemaEditor
    ) -> str | None:
        if self.condition is None:
            return None
        query = Query(model=model, alias_cols=False)
        where = query.build_where(self.condition)
        compiler = query.get_compiler()
        sql, params = where.as_sql(compiler, schema_editor.connection)
        return sql % tuple(schema_editor.quote_value(p) for p in params)

    def _get_index_expressions(
        self, model: type[Model], schema_editor: BaseDatabaseSchemaEditor
    ) -> Any:
        if not self.expressions:
            return None
        index_expressions = []
        for expression in self.expressions:
            index_expression = IndexExpression(expression)
            index_expression.set_wrapper_classes(schema_editor.connection)
            index_expressions.append(index_expression)
        return ExpressionList(*index_expressions).resolve_expression(
            Query(model, alias_cols=False),
        )

    def constraint_sql(
        self, model: type[Model], schema_editor: BaseDatabaseSchemaEditor
    ) -> str | None:
        fields = [model._model_meta.get_field(field_name) for field_name in self.fields]
        include = [
            model._model_meta.get_field(field_name).column
            for field_name in self.include
        ]
        condition = self._get_condition_sql(model, schema_editor)
        expressions = self._get_index_expressions(model, schema_editor)
        return schema_editor._unique_sql(
            model,
            fields,
            self.name,
            condition=condition,
            deferrable=self.deferrable,
            include=include,
            opclasses=tuple(self.opclasses) if self.opclasses else None,
            expressions=expressions,
        )

    def create_sql(
        self, model: type[Model], schema_editor: BaseDatabaseSchemaEditor
    ) -> Statement | None:
        fields = [model._model_meta.get_field(field_name) for field_name in self.fields]
        include = [
            model._model_meta.get_field(field_name).column
            for field_name in self.include
        ]
        condition = self._get_condition_sql(model, schema_editor)
        expressions = self._get_index_expressions(model, schema_editor)
        return schema_editor._create_unique_sql(
            model,
            fields,
            self.name,
            condition=condition,
            deferrable=self.deferrable,
            include=include,
            opclasses=tuple(self.opclasses) if self.opclasses else None,
            expressions=expressions,
        )

    def remove_sql(
        self, model: type[Model], schema_editor: BaseDatabaseSchemaEditor
    ) -> Statement | None:
        condition = self._get_condition_sql(model, schema_editor)
        include = [
            model._model_meta.get_field(field_name).column
            for field_name in self.include
        ]
        expressions = self._get_index_expressions(model, schema_editor)
        return schema_editor._delete_unique_sql(
            model,
            self.name,
            condition=condition,
            deferrable=self.deferrable,
            include=include,
            opclasses=tuple(self.opclasses) if self.opclasses else None,
            expressions=expressions,
        )

    def __repr__(self) -> str:
        return "<{}:{}{}{}{}{}{}{}{}{}>".format(
            self.__class__.__qualname__,
            "" if not self.fields else f" fields={repr(self.fields)}",
            "" if not self.expressions else f" expressions={repr(self.expressions)}",
            f" name={repr(self.name)}",
            "" if self.condition is None else f" condition={self.condition}",
            "" if self.deferrable is None else f" deferrable={self.deferrable!r}",
            "" if not self.include else f" include={repr(self.include)}",
            "" if not self.opclasses else f" opclasses={repr(self.opclasses)}",
            (
                ""
                if self.violation_error_code is None
                else f" violation_error_code={self.violation_error_code!r}"
            ),
            (
                ""
                if self.violation_error_message is None
                or self.violation_error_message == self.default_violation_error_message
                else f" violation_error_message={self.violation_error_message!r}"
            ),
        )

    def __eq__(self, other: object) -> bool:
        if isinstance(other, UniqueConstraint):
            return (
                self.name == other.name
                and self.fields == other.fields
                and self.condition == other.condition
                and self.deferrable == other.deferrable
                and self.include == other.include
                and self.opclasses == other.opclasses
                and self.expressions == other.expressions
                and self.violation_error_code == other.violation_error_code
                and self.violation_error_message == other.violation_error_message
            )
        return super().__eq__(other)

    def deconstruct(self) -> tuple[str, tuple[Any, ...], dict[str, Any]]:
        path, args, kwargs = super().deconstruct()
        if self.fields:
            kwargs["fields"] = self.fields
        if self.condition:
            kwargs["condition"] = self.condition
        if self.deferrable:
            kwargs["deferrable"] = self.deferrable
        if self.include:
            kwargs["include"] = self.include
        if self.opclasses:
            kwargs["opclasses"] = self.opclasses
        return path, self.expressions, kwargs

    def validate(
        self, model: type[Model], instance: Model, exclude: set[str] | None = None
    ) -> None:
        queryset = model.query
        if self.fields:
            lookup_kwargs = {}
            for field_name in self.fields:
                if exclude and field_name in exclude:
                    return
                field = model._model_meta.get_field(field_name)
                lookup_value = getattr(instance, field.attname)
                if lookup_value is None:
                    # A composite constraint containing NULL value cannot cause
                    # a violation since NULL != NULL in SQL.
                    return
                lookup_kwargs[field.name] = lookup_value
            queryset = queryset.filter(**lookup_kwargs)
        else:
            # Ignore constraints with excluded fields.
            if exclude:
                for expression in self.expressions:
                    if hasattr(expression, "flatten"):
                        for expr in expression.flatten():
                            if isinstance(expr, F) and expr.name in exclude:
                                return
                    elif isinstance(expression, F) and expression.name in exclude:
                        return
            replacements = {
                F(field): value
                for field, value in instance._get_field_value_map(
                    meta=model._model_meta, exclude=exclude
                ).items()
            }
            expressions = []
            for expr in self.expressions:
                # Ignore ordering.
                if isinstance(expr, OrderBy):
                    expr = expr.expression
                expressions.append(Exact(expr, expr.replace_expressions(replacements)))
            queryset = queryset.filter(*expressions)
        model_class_id = instance.id
        if not instance._state.adding and model_class_id is not None:
            queryset = queryset.exclude(id=model_class_id)
        if not self.condition:
            if queryset.exists():
                if self.expressions:
                    raise ValidationError(
                        self.get_violation_error_message(),
                        code=self.violation_error_code,
                    )
                # When fields are defined, use the unique_error_message() for
                # backward compatibility.
                for constraint_model, constraints in instance.get_constraints():
                    for constraint in constraints:
                        if constraint is self:
                            raise ValidationError(
                                instance.unique_error_message(
                                    constraint_model,  # type: ignore[arg-type]
                                    self.fields,
                                ),
                            )
        else:
            against = instance._get_field_value_map(
                meta=model._model_meta, exclude=exclude
            )
            try:
                if (self.condition & Exists(queryset.filter(self.condition))).check(
                    against
                ):
                    raise ValidationError(
                        self.get_violation_error_message(),
                        code=self.violation_error_code,
                    )
            except FieldError:
                pass
