from __future__ import annotations

import os
import sys
from typing import TYPE_CHECKING, Any

from plain.runtime import settings

if TYPE_CHECKING:
    from plain.models.backends.base.base import BaseDatabaseWrapper

# The prefix to put on the default database name when creating
# the test database.
TEST_DATABASE_PREFIX = "test_"


class BaseDatabaseCreation:
    """
    Encapsulate backend-specific differences pertaining to creation and
    destruction of the test database.
    """

    def __init__(self, connection: BaseDatabaseWrapper):
        self.connection = connection

    def _nodb_cursor(self) -> Any:
        return self.connection._nodb_cursor()

    def log(self, msg: str) -> None:
        sys.stderr.write(msg + os.linesep)

    def create_test_db(self, verbosity: int = 1, prefix: str = "") -> str:
        """
        Create a test database, prompting the user for confirmation if the
        database already exists. Return the name of the test database created.

        If prefix is provided, it will be prepended to the database name
        to isolate it from other test databases.
        """
        from plain.models.cli.migrations import apply

        test_database_name = self._get_test_db_name(prefix)

        if verbosity >= 1:
            self.log(f"Creating test database '{test_database_name}'...")

        self._create_test_db(
            test_database_name=test_database_name, verbosity=verbosity, autoclobber=True
        )

        self.connection.close()
        settings.DATABASE["NAME"] = test_database_name
        self.connection.settings_dict["NAME"] = test_database_name

        apply.callback(
            package_label=None,
            migration_name=None,
            fake=False,
            plan=False,
            check_unapplied=False,
            backup=False,
            no_input=True,
            atomic_batch=False,  # No need for atomic batch when creating test database
            quiet=verbosity < 2,  # Show migration output when verbosity is 2+
        )

        # Ensure a connection for the side effect of initializing the test database.
        self.connection.ensure_connection()

        return test_database_name

    def set_as_test_mirror(self, primary_settings_dict: dict[str, Any]) -> None:
        """
        Set this database up to be used in testing as a mirror of a primary
        database whose settings are given.
        """
        self.connection.settings_dict["NAME"] = primary_settings_dict["NAME"]

    # def serialize_db_to_string(self):
    #     """
    #     Serialize all data in the database into a JSON string.
    #     Designed only for test runner usage; will not handle large
    #     amounts of data.
    #     """

    #     # Iteratively return every object for all models to serialize.
    #     def get_objects():
    #         from plain.models.migrations.loader import MigrationLoader

    #         loader = MigrationLoader(self.connection)
    #         for package_config in packages.get_package_configs():
    #             if (
    #                 package_config.models_module is not None
    #                 and package_config.package_label in loader.migrated_packages
    #             ):
    #                 for model in package_config.get_models():
    #                     if model.model_options.can_migrate(
    #                         self.connection
    #                     ) and router.allow_migrate_model(self.connection.alias, model):
    #                         queryset = model._base_manager.using(
    #                             self.connection.alias,
    #                         ).order_by("id")
    #                         yield from queryset.iterator()

    #     # Serialize to a string
    #     out = StringIO()
    #     serializers.serialize("json", get_objects(), indent=None, stream=out)
    #     return out.getvalue()

    # def deserialize_db_from_string(self, data):
    #     """
    #     Reload the database with data from a string generated by
    #     the serialize_db_to_string() method.
    #     """
    #     data = StringIO(data)
    #     table_names = set()
    #     # Load data in a transaction to handle forward references and cycles.
    #     with atomic(using=self.connection.alias):
    #         # Disable constraint checks, because some databases (MySQL) doesn't
    #         # support deferred checks.
    #         with self.connection.constraint_checks_disabled():
    #             for obj in serializers.deserialize(
    #                 "json", data, using=self.connection.alias
    #             ):
    #                 obj.save()
    #                 table_names.add(obj.object.model_options.db_table)
    #         # Manually check for any invalid keys that might have been added,
    #         # because constraint checks were disabled.
    #         self.connection.check_constraints(table_names=table_names)

    def _get_test_db_name(self, prefix: str = "") -> str:
        """
        Internal implementation - return the name of the test DB that will be
        created. Only useful when called from create_test_db() and
        _create_test_db() and when no external munging is done with the 'NAME'
        settings.

        If prefix is provided, it will be prepended to the database name.
        """
        # Determine the base name: explicit TEST.NAME overrides base NAME.
        base_name = (
            self.connection.settings_dict["TEST"]["NAME"]
            or self.connection.settings_dict["NAME"]
        )
        if prefix:
            return f"{prefix}_{base_name}"
        if self.connection.settings_dict["TEST"]["NAME"]:
            return self.connection.settings_dict["TEST"]["NAME"]
        return TEST_DATABASE_PREFIX + self.connection.settings_dict["NAME"]

    def _execute_create_test_db(self, cursor: Any, parameters: dict[str, str]) -> None:
        cursor.execute("CREATE DATABASE {dbname} {suffix}".format(**parameters))

    def _create_test_db(
        self, *, test_database_name: str, verbosity: int, autoclobber: bool
    ) -> str:
        """
        Internal implementation - create the test db tables.
        """
        test_db_params = {
            "dbname": self.connection.ops.quote_name(test_database_name),
            "suffix": self.sql_table_creation_suffix(),
        }
        # Create the test database and connect to it.
        with self._nodb_cursor() as cursor:
            try:
                self._execute_create_test_db(cursor, test_db_params)
            except Exception as e:
                self.log(f"Got an error creating the test database: {e}")
                if not autoclobber:
                    confirm = input(
                        "Type 'yes' if you would like to try deleting the test "
                        f"database '{test_database_name}', or 'no' to cancel: "
                    )
                if autoclobber or confirm == "yes":
                    try:
                        if verbosity >= 1:
                            self.log(
                                f"Destroying old test database '{test_database_name}'..."
                            )
                        cursor.execute(
                            "DROP DATABASE {dbname}".format(**test_db_params)
                        )
                        self._execute_create_test_db(cursor, test_db_params)
                    except Exception as e:
                        self.log(f"Got an error recreating the test database: {e}")
                        sys.exit(2)
                else:
                    self.log("Tests cancelled.")
                    sys.exit(1)

        return test_database_name

    def destroy_test_db(
        self, old_database_name: str | None = None, verbosity: int = 1
    ) -> None:
        """
        Destroy a test database, prompting the user for confirmation if the
        database already exists.
        """
        self.connection.close()

        test_database_name = self.connection.settings_dict["NAME"]

        if verbosity >= 1:
            self.log(f"Destroying test database '{test_database_name}'...")
        self._destroy_test_db(test_database_name, verbosity)

        # Restore the original database name
        if old_database_name is not None:
            settings.DATABASE["NAME"] = old_database_name
            self.connection.settings_dict["NAME"] = old_database_name

    def _destroy_test_db(self, test_database_name: str, verbosity: int) -> None:
        """
        Internal implementation - remove the test db tables.
        """
        # Remove the test database to clean up after
        # ourselves. Connect to the previous database (not the test database)
        # to do so, because it's not allowed to delete a database while being
        # connected to it.
        with self._nodb_cursor() as cursor:
            cursor.execute(
                f"DROP DATABASE {self.connection.ops.quote_name(test_database_name)}"
            )

    def sql_table_creation_suffix(self) -> str:
        """
        SQL to append to the end of the test table creation statements.
        """
        return ""

    def test_db_signature(self, prefix: str = "") -> tuple[str, str, str, str]:
        """
        Return a tuple with elements of self.connection.settings_dict (a
        DATABASE setting value) that uniquely identify a database
        accordingly to the RDBMS particularities.
        """
        settings_dict = self.connection.settings_dict
        return (
            settings_dict["HOST"],
            settings_dict["PORT"],
            settings_dict["ENGINE"],
            self._get_test_db_name(prefix),
        )
