from __future__ import annotations

import _thread
import copy
import datetime
import logging
import threading
import time
import warnings
import zoneinfo
from abc import ABC, abstractmethod
from collections import deque
from collections.abc import Generator
from contextlib import contextmanager
from functools import cached_property
from typing import TYPE_CHECKING, Any

from plain.models.backends import utils
from plain.models.backends.base.validation import BaseDatabaseValidation
from plain.models.backends.utils import debug_transaction
from plain.models.db import (
    DatabaseError,
    DatabaseErrorWrapper,
    NotSupportedError,
)
from plain.models.transaction import TransactionManagementError
from plain.runtime import settings

if TYPE_CHECKING:
    from plain.models.backends.base.client import BaseDatabaseClient
    from plain.models.backends.base.creation import BaseDatabaseCreation
    from plain.models.backends.base.features import BaseDatabaseFeatures
    from plain.models.backends.base.introspection import BaseDatabaseIntrospection
    from plain.models.backends.base.operations import BaseDatabaseOperations
    from plain.models.backends.base.schema import BaseDatabaseSchemaEditor

RAN_DB_VERSION_CHECK = False

logger = logging.getLogger("plain.models.backends.base")


class BaseDatabaseWrapper(ABC):
    """Represent a database connection."""

    # Mapping of Field objects to their column types.
    data_types: dict[str, str] = {}
    # Mapping of Field objects to their SQL suffix such as AUTOINCREMENT.
    data_types_suffix: dict[str, str] = {}
    # Mapping of Field objects to their SQL for CHECK constraints.
    data_type_check_constraints: dict[str, str] = {}
    # Instance attributes - always assigned in __init__
    ops: BaseDatabaseOperations
    client: BaseDatabaseClient
    creation: BaseDatabaseCreation
    features: BaseDatabaseFeatures
    introspection: BaseDatabaseIntrospection
    validation: BaseDatabaseValidation
    vendor: str = "unknown"
    display_name: str = "unknown"
    SchemaEditorClass: type[BaseDatabaseSchemaEditor] | None = None
    # Classes instantiated in __init__().
    client_class: type[BaseDatabaseClient] | None = None
    creation_class: type[BaseDatabaseCreation] | None = None
    features_class: type[BaseDatabaseFeatures] | None = None
    introspection_class: type[BaseDatabaseIntrospection] | None = None
    ops_class: type[BaseDatabaseOperations] | None = None
    validation_class: type[BaseDatabaseValidation] = BaseDatabaseValidation
    Database: Any

    queries_limit: int = 9000

    def __init__(self, settings_dict: dict[str, Any]):
        # Connection related attributes.
        # The underlying database connection (from the database library, not a wrapper).
        self.connection: Any = None
        # `settings_dict` should be a dictionary containing keys such as
        # NAME, USER, etc. It's called `settings_dict` instead of `settings`
        # to disambiguate it from Plain settings modules.
        self.settings_dict: dict[str, Any] = settings_dict
        # Query logging in debug mode or when explicitly enabled.
        self.queries_log: deque[dict[str, Any]] = deque(maxlen=self.queries_limit)
        self.force_debug_cursor: bool = False

        # Transaction related attributes.
        # Tracks if the connection is in autocommit mode. Per PEP 249, by
        # default, it isn't.
        self.autocommit: bool = False
        # Tracks if the connection is in a transaction managed by 'atomic'.
        self.in_atomic_block: bool = False
        # Increment to generate unique savepoint ids.
        self.savepoint_state: int = 0
        # List of savepoints created by 'atomic'.
        self.savepoint_ids: list[str] = []
        # Stack of active 'atomic' blocks.
        self.atomic_blocks: list[Any] = []
        # Tracks if the outermost 'atomic' block should commit on exit,
        # ie. if autocommit was active on entry.
        self.commit_on_exit: bool = True
        # Tracks if the transaction should be rolled back to the next
        # available savepoint because of an exception in an inner block.
        self.needs_rollback: bool = False
        self.rollback_exc: Exception | None = None

        # Connection termination related attributes.
        self.close_at: float | None = None
        self.closed_in_transaction: bool = False
        self.errors_occurred: bool = False
        self.health_check_enabled: bool = False
        self.health_check_done: bool = False

        # Thread-safety related attributes.
        self._thread_sharing_lock: threading.Lock = threading.Lock()
        self._thread_sharing_count: int = 0
        self._thread_ident: int = _thread.get_ident()

        # A list of no-argument functions to run when the transaction commits.
        # Each entry is an (sids, func, robust) tuple, where sids is a set of
        # the active savepoint IDs when this function was registered and robust
        # specifies whether it's allowed for the function to fail.
        self.run_on_commit: list[tuple[set[str], Any, bool]] = []

        # Should we run the on-commit hooks the next time set_autocommit(True)
        # is called?
        self.run_commit_hooks_on_set_autocommit_on: bool = False

        # A stack of wrappers to be invoked around execute()/executemany()
        # calls. Each entry is a function taking five arguments: execute, sql,
        # params, many, and context. It's the function's responsibility to
        # call execute(sql, params, many, context).
        self.execute_wrappers: list[Any] = []

        self.client: BaseDatabaseClient = self.client_class(self)  # type: ignore[misc]
        self.creation: BaseDatabaseCreation = self.creation_class(self)  # type: ignore[misc]
        self.features: BaseDatabaseFeatures = self.features_class(self)  # type: ignore[misc]
        self.introspection: BaseDatabaseIntrospection = self.introspection_class(self)  # type: ignore[misc]
        self.ops: BaseDatabaseOperations = self.ops_class(self)  # type: ignore[misc]
        self.validation: BaseDatabaseValidation = self.validation_class(self)

    def __repr__(self) -> str:
        return f"<{self.__class__.__qualname__} vendor={self.vendor!r}>"

    def ensure_timezone(self) -> bool:
        """
        Ensure the connection's timezone is set to `self.timezone_name` and
        return whether it changed or not.
        """
        return False

    @cached_property
    def timezone(self) -> datetime.tzinfo:
        """
        Return a tzinfo of the database connection time zone.

        This is only used when time zone support is enabled. When a datetime is
        read from the database, it is always returned in this time zone.

        When the database backend supports time zones, it doesn't matter which
        time zone Plain uses, as long as aware datetimes are used everywhere.
        Other users connecting to the database can choose their own time zone.

        When the database backend doesn't support time zones, the time zone
        Plain uses may be constrained by the requirements of other users of
        the database.
        """
        if self.settings_dict["TIME_ZONE"] is None:
            return datetime.UTC
        else:
            return zoneinfo.ZoneInfo(self.settings_dict["TIME_ZONE"])

    @cached_property
    def timezone_name(self) -> str:
        """
        Name of the time zone of the database connection.
        """
        if self.settings_dict["TIME_ZONE"] is None:
            return "UTC"
        else:
            return self.settings_dict["TIME_ZONE"]

    @property
    def queries_logged(self) -> bool:
        return self.force_debug_cursor or settings.DEBUG

    @property
    def queries(self) -> list[dict[str, Any]]:
        if len(self.queries_log) == self.queries_log.maxlen:
            warnings.warn(
                f"Limit for query logging exceeded, only the last {self.queries_log.maxlen} queries "
                "will be returned."
            )
        return list(self.queries_log)

    @abstractmethod
    def get_database_version(self) -> tuple[int, ...]:
        """Return a tuple of the database's version."""
        ...

    def check_database_version_supported(self) -> None:
        """
        Raise an error if the database version isn't supported by this
        version of Plain.
        """
        if (
            self.features.minimum_database_version is not None
            and self.get_database_version() < self.features.minimum_database_version
        ):
            db_version = ".".join(map(str, self.get_database_version()))
            min_db_version = ".".join(map(str, self.features.minimum_database_version))
            raise NotSupportedError(
                f"{self.display_name} {min_db_version} or later is required "
                f"(found {db_version})."
            )

    # ##### Backend-specific methods for creating connections and cursors #####

    @abstractmethod
    def get_connection_params(self) -> dict[str, Any]:
        """Return a dict of parameters suitable for get_new_connection."""
        ...

    @abstractmethod
    def get_new_connection(self, conn_params: dict[str, Any]) -> Any:
        """Open a connection to the database."""
        ...

    def init_connection_state(self) -> None:
        """Initialize the database connection settings."""
        global RAN_DB_VERSION_CHECK
        if not RAN_DB_VERSION_CHECK:
            self.check_database_version_supported()
            RAN_DB_VERSION_CHECK = True

    @abstractmethod
    def create_cursor(self, name: str | None = None) -> Any:
        """Create a cursor. Assume that a connection is established."""
        ...

    # ##### Backend-specific methods for creating connections #####

    def connect(self) -> None:
        """Connect to the database. Assume that the connection is closed."""
        # In case the previous connection was closed while in an atomic block
        self.in_atomic_block = False
        self.savepoint_ids = []
        self.atomic_blocks = []
        self.needs_rollback = False
        # Reset parameters defining when to close/health-check the connection.
        self.health_check_enabled = self.settings_dict["CONN_HEALTH_CHECKS"]
        max_age = self.settings_dict["CONN_MAX_AGE"]
        self.close_at = None if max_age is None else time.monotonic() + max_age
        self.closed_in_transaction = False
        self.errors_occurred = False
        # New connections are healthy.
        self.health_check_done = True
        # Establish the connection
        conn_params = self.get_connection_params()
        self.connection = self.get_new_connection(conn_params)
        self.set_autocommit(self.settings_dict["AUTOCOMMIT"])
        self.init_connection_state()

        self.run_on_commit = []

    def ensure_connection(self) -> None:
        """Guarantee that a connection to the database is established."""
        if self.connection is None:
            with self.wrap_database_errors:
                self.connect()

    # ##### Backend-specific wrappers for PEP-249 connection methods #####

    def _prepare_cursor(self, cursor: Any) -> utils.CursorWrapper:
        """
        Validate the connection is usable and perform database cursor wrapping.
        """
        self.validate_thread_sharing()
        if self.queries_logged:
            wrapped_cursor = self.make_debug_cursor(cursor)
        else:
            wrapped_cursor = self.make_cursor(cursor)
        return wrapped_cursor

    def _cursor(self, name: str | None = None) -> utils.CursorWrapper:
        self.close_if_health_check_failed()
        self.ensure_connection()
        with self.wrap_database_errors:
            return self._prepare_cursor(self.create_cursor(name))

    def _commit(self) -> None:
        if self.connection is not None:
            with debug_transaction(self, "COMMIT"), self.wrap_database_errors:
                return self.connection.commit()

    def _rollback(self) -> None:
        if self.connection is not None:
            with debug_transaction(self, "ROLLBACK"), self.wrap_database_errors:
                return self.connection.rollback()

    def _close(self) -> None:
        if self.connection is not None:
            with self.wrap_database_errors:
                return self.connection.close()

    # ##### Generic wrappers for PEP-249 connection methods #####

    def cursor(self) -> utils.CursorWrapper:
        """Create a cursor, opening a connection if necessary."""
        return self._cursor()

    def commit(self) -> None:
        """Commit a transaction and reset the dirty flag."""
        self.validate_thread_sharing()
        self.validate_no_atomic_block()
        self._commit()
        # A successful commit means that the database connection works.
        self.errors_occurred = False
        self.run_commit_hooks_on_set_autocommit_on = True

    def rollback(self) -> None:
        """Roll back a transaction and reset the dirty flag."""
        self.validate_thread_sharing()
        self.validate_no_atomic_block()
        self._rollback()
        # A successful rollback means that the database connection works.
        self.errors_occurred = False
        self.needs_rollback = False
        self.run_on_commit = []

    def close(self) -> None:
        """Close the connection to the database."""
        self.validate_thread_sharing()
        self.run_on_commit = []

        # Don't call validate_no_atomic_block() to avoid making it difficult
        # to get rid of a connection in an invalid state. The next connect()
        # will reset the transaction state anyway.
        if self.closed_in_transaction or self.connection is None:
            return
        try:
            self._close()
        finally:
            if self.in_atomic_block:
                self.closed_in_transaction = True
                self.needs_rollback = True
            else:
                self.connection = None

    # ##### Backend-specific savepoint management methods #####

    def _savepoint(self, sid: str) -> None:
        with self.cursor() as cursor:
            cursor.execute(self.ops.savepoint_create_sql(sid))

    def _savepoint_rollback(self, sid: str) -> None:
        with self.cursor() as cursor:
            cursor.execute(self.ops.savepoint_rollback_sql(sid))

    def _savepoint_commit(self, sid: str) -> None:
        with self.cursor() as cursor:
            cursor.execute(self.ops.savepoint_commit_sql(sid))

    def _savepoint_allowed(self) -> bool:
        # Savepoints cannot be created outside a transaction
        return self.features.uses_savepoints and not self.get_autocommit()

    # ##### Generic savepoint management methods #####

    def savepoint(self) -> str | None:
        """
        Create a savepoint inside the current transaction. Return an
        identifier for the savepoint that will be used for the subsequent
        rollback or commit. Do nothing if savepoints are not supported.
        """
        if not self._savepoint_allowed():
            return None

        thread_ident = _thread.get_ident()
        tid = str(thread_ident).replace("-", "")

        self.savepoint_state += 1
        sid = "s%s_x%d" % (tid, self.savepoint_state)  # noqa: UP031

        self.validate_thread_sharing()
        self._savepoint(sid)

        return sid

    def savepoint_rollback(self, sid: str) -> None:
        """
        Roll back to a savepoint. Do nothing if savepoints are not supported.
        """
        if not self._savepoint_allowed():
            return

        self.validate_thread_sharing()
        self._savepoint_rollback(sid)

        # Remove any callbacks registered while this savepoint was active.
        self.run_on_commit = [
            (sids, func, robust)
            for (sids, func, robust) in self.run_on_commit
            if sid not in sids
        ]

    def savepoint_commit(self, sid: str) -> None:
        """
        Release a savepoint. Do nothing if savepoints are not supported.
        """
        if not self._savepoint_allowed():
            return

        self.validate_thread_sharing()
        self._savepoint_commit(sid)

    def clean_savepoints(self) -> None:
        """
        Reset the counter used to generate unique savepoint ids in this thread.
        """
        self.savepoint_state = 0

    # ##### Backend-specific transaction management methods #####

    @abstractmethod
    def _set_autocommit(self, autocommit: bool) -> None:
        """
        Backend-specific implementation to enable or disable autocommit.
        """
        ...

    # ##### Generic transaction management methods #####

    def get_autocommit(self) -> bool:
        """Get the autocommit state."""
        self.ensure_connection()
        return self.autocommit

    def set_autocommit(
        self,
        autocommit: bool,
        force_begin_transaction_with_broken_autocommit: bool = False,
    ) -> None:
        """
        Enable or disable autocommit.

        The usual way to start a transaction is to turn autocommit off.
        SQLite does not properly start a transaction when disabling
        autocommit. To avoid this buggy behavior and to actually enter a new
        transaction, an explicit BEGIN is required. Using
        force_begin_transaction_with_broken_autocommit=True will issue an
        explicit BEGIN with SQLite. This option will be ignored for other
        backends.
        """
        self.validate_no_atomic_block()
        self.close_if_health_check_failed()
        self.ensure_connection()

        start_transaction_under_autocommit = (
            force_begin_transaction_with_broken_autocommit
            and not autocommit
            and hasattr(self, "_start_transaction_under_autocommit")
        )

        if start_transaction_under_autocommit:
            self._start_transaction_under_autocommit()  # type: ignore[attr-defined]
        elif autocommit:
            self._set_autocommit(autocommit)
        else:
            with debug_transaction(self, "BEGIN"):
                self._set_autocommit(autocommit)
        self.autocommit = autocommit

        if autocommit and self.run_commit_hooks_on_set_autocommit_on:
            self.run_and_clear_commit_hooks()
            self.run_commit_hooks_on_set_autocommit_on = False

    def get_rollback(self) -> bool:
        """Get the "needs rollback" flag -- for *advanced use* only."""
        if not self.in_atomic_block:
            raise TransactionManagementError(
                "The rollback flag doesn't work outside of an 'atomic' block."
            )
        return self.needs_rollback

    def set_rollback(self, rollback: bool) -> None:
        """
        Set or unset the "needs rollback" flag -- for *advanced use* only.
        """
        if not self.in_atomic_block:
            raise TransactionManagementError(
                "The rollback flag doesn't work outside of an 'atomic' block."
            )
        self.needs_rollback = rollback

    def validate_no_atomic_block(self) -> None:
        """Raise an error if an atomic block is active."""
        if self.in_atomic_block:
            raise TransactionManagementError(
                "This is forbidden when an 'atomic' block is active."
            )

    def validate_no_broken_transaction(self) -> None:
        if self.needs_rollback:
            raise TransactionManagementError(
                "An error occurred in the current transaction. You can't "
                "execute queries until the end of the 'atomic' block."
            ) from self.rollback_exc

    # ##### Foreign key constraints checks handling #####

    def disable_constraint_checking(self) -> bool:
        """
        Backends can implement as needed to temporarily disable foreign key
        constraint checking. Should return True if the constraints were
        disabled and will need to be reenabled.
        """
        return False

    def enable_constraint_checking(self) -> None:
        """
        Backends can implement as needed to re-enable foreign key constraint
        checking.
        """
        pass

    def check_constraints(self, table_names: list[str] | None = None) -> None:
        """
        Backends can override this method if they can apply constraint
        checking (e.g. via "SET CONSTRAINTS ALL IMMEDIATE"). Should raise an
        IntegrityError if any invalid foreign key references are encountered.
        """
        pass

    # ##### Connection termination handling #####

    @abstractmethod
    def is_usable(self) -> bool:
        """
        Test if the database connection is usable.

        This method may assume that self.connection is not None.

        Actual implementations should take care not to raise exceptions
        as that may prevent Plain from recycling unusable connections.
        """
        ...

    def close_if_health_check_failed(self) -> None:
        """Close existing connection if it fails a health check."""
        if (
            self.connection is None
            or not self.health_check_enabled
            or self.health_check_done
        ):
            return

        if not self.is_usable():
            self.close()
        self.health_check_done = True

    def close_if_unusable_or_obsolete(self) -> None:
        """
        Close the current connection if unrecoverable errors have occurred
        or if it outlived its maximum age.
        """
        if self.connection is not None:
            self.health_check_done = False
            # If the application didn't restore the original autocommit setting,
            # don't take chances, drop the connection.
            if self.get_autocommit() != self.settings_dict["AUTOCOMMIT"]:
                self.close()
                return

            # If an exception other than DataError or IntegrityError occurred
            # since the last commit / rollback, check if the connection works.
            if self.errors_occurred:
                if self.is_usable():
                    self.errors_occurred = False
                    self.health_check_done = True
                else:
                    self.close()
                    return

            if self.close_at is not None and time.monotonic() >= self.close_at:
                self.close()
                return

    # ##### Thread safety handling #####

    @property
    def allow_thread_sharing(self) -> bool:
        with self._thread_sharing_lock:
            return self._thread_sharing_count > 0

    def validate_thread_sharing(self) -> None:
        """
        Validate that the connection isn't accessed by another thread than the
        one which originally created it, unless the connection was explicitly
        authorized to be shared between threads (via the `inc_thread_sharing()`
        method). Raise an exception if the validation fails.
        """
        if not (self.allow_thread_sharing or self._thread_ident == _thread.get_ident()):
            raise DatabaseError(
                "DatabaseWrapper objects created in a "
                "thread can only be used in that same thread. The connection "
                f"was created in thread id {self._thread_ident} and this is "
                f"thread id {_thread.get_ident()}."
            )

    # ##### Miscellaneous #####

    def prepare_database(self) -> None:
        """
        Hook to do any database check or preparation, generally called before
        migrating a project or an app.
        """
        pass

    @cached_property
    def wrap_database_errors(self) -> DatabaseErrorWrapper:
        """
        Context manager and decorator that re-throws backend-specific database
        exceptions using Plain's common wrappers.
        """
        return DatabaseErrorWrapper(self)

    def chunked_cursor(self) -> utils.CursorWrapper:
        """
        Return a cursor that tries to avoid caching in the database (if
        supported by the database), otherwise return a regular cursor.
        """
        return self.cursor()

    def make_debug_cursor(self, cursor: Any) -> utils.CursorDebugWrapper:
        """Create a cursor that logs all queries in self.queries_log."""
        return utils.CursorDebugWrapper(cursor, self)

    def make_cursor(self, cursor: Any) -> utils.CursorWrapper:
        """Create a cursor without debug logging."""
        return utils.CursorWrapper(cursor, self)

    @contextmanager
    def temporary_connection(self) -> Generator[utils.CursorWrapper, None, None]:
        """
        Context manager that ensures that a connection is established, and
        if it opened one, closes it to avoid leaving a dangling connection.
        This is useful for operations outside of the request-response cycle.

        Provide a cursor: with self.temporary_connection() as cursor: ...
        """
        must_close = self.connection is None
        try:
            with self.cursor() as cursor:
                yield cursor
        finally:
            if must_close:
                self.close()

    @contextmanager
    def _nodb_cursor(self) -> Generator[utils.CursorWrapper, None, None]:
        """
        Return a cursor from an alternative connection to be used when there is
        no need to access the main database, specifically for test db
        creation/deletion. This also prevents the production database from
        being exposed to potential child threads while (or after) the test
        database is destroyed. Refs #10868, #17786, #16969.
        """
        conn = self.__class__({**self.settings_dict, "NAME": None})
        try:
            with conn.cursor() as cursor:
                yield cursor
        finally:
            conn.close()

    def schema_editor(self, *args: Any, **kwargs: Any) -> BaseDatabaseSchemaEditor:
        """
        Return a new instance of this backend's SchemaEditor.
        """
        if self.SchemaEditorClass is None:
            raise NotImplementedError(
                "The SchemaEditorClass attribute of this database wrapper is still None"
            )
        return self.SchemaEditorClass(self, *args, **kwargs)

    def on_commit(self, func: Any, robust: bool = False) -> None:
        if not callable(func):
            raise TypeError("on_commit()'s callback must be a callable.")
        if self.in_atomic_block:
            # Transaction in progress; save for execution on commit.
            self.run_on_commit.append((set(self.savepoint_ids), func, robust))
        elif not self.get_autocommit():
            raise TransactionManagementError(
                "on_commit() cannot be used in manual transaction management"
            )
        else:
            # No transaction in progress and in autocommit mode; execute
            # immediately.
            if robust:
                try:
                    func()
                except Exception as e:
                    logger.error(
                        f"Error calling {func.__qualname__} in on_commit() (%s).",
                        e,
                        exc_info=True,
                    )
            else:
                func()

    def run_and_clear_commit_hooks(self) -> None:
        self.validate_no_atomic_block()
        current_run_on_commit = self.run_on_commit
        self.run_on_commit = []
        while current_run_on_commit:
            _, func, robust = current_run_on_commit.pop(0)
            if robust:
                try:
                    func()
                except Exception as e:
                    logger.error(
                        f"Error calling {func.__qualname__} in on_commit() during "
                        f"transaction (%s).",
                        e,
                        exc_info=True,
                    )
            else:
                func()

    @contextmanager
    def execute_wrapper(self, wrapper: Any) -> Generator[None, None, None]:
        """
        Return a context manager under which the wrapper is applied to suitable
        database query executions.
        """
        self.execute_wrappers.append(wrapper)
        try:
            yield
        finally:
            self.execute_wrappers.pop()

    def copy(self) -> BaseDatabaseWrapper:
        """
        Return a copy of this connection.

        For tests that require two connections to the same database.
        """
        settings_dict = copy.deepcopy(self.settings_dict)
        return type(self)(settings_dict)
