
from typing import List
from pyrdfrules.common.logging.logger import log
from pyrdfrules.common.result.evaluation import Evaluation
from pyrdfrules.common.result.histogram import HistogramResult, HistogramSingleResult
from pyrdfrules.common.rule.resultrule import ResultRule
from pyrdfrules.common.rule.ruleset import Ruleset

class Result():
    """Class representing the result of a task.
    """
    
    ruleset: Ruleset = None
    """Ruleset generated by RDFRules.
    """
    
    evaluate: List[Evaluation] = []
    
    predictionTasks: List[dict] = None
    
    histogram: HistogramResult = None
    
    data: dict
    """Raw JSON response from RDFRules."""
    
    def __init__(self, data: dict) -> None:
        """Creates a new Result object.

        Args:
            data (dict): JSON response from RDFRules.
        """
        self.data = data
        
        self._parse_data()
        
        pass
    
    def _parse_data(self):
        """Internal function translating the resulting JSON into internal object representations.
        """
        
        rules = []
        predictionTasks = []
        evaluate = []
        histogram = []
        
        for item in self.data:
            log().debug(f"Parsing item: {str(item)}")

            match item:
                case {'body': _, 'head': __, 'measures': ___}:
                    # Item is a rule
                    rule = ResultRule.model_validate(item)
                    rules.append(rule)               
                    pass
                
                case {'candidates': _, 'predictionTask': __}:
                    # Item is a prediction task
                    predictionTasks.append(item)
                    pass
                
                case {'error': _}:
                    # Item is an error
                    print("Error: " + item['error'])
                    pass
                
                case {'hits': _, 'type': 'ranking'}:
                    # Item is an evaluation
                    evaluate.append(Evaluation.model_validate(item))
                    pass
                
                case {'fp': _, 'type': 'completeness'}:
                    # Item is an evaluation
                    evaluate.append(Evaluation.model_validate(item))
                    pass
                
                case {'n': _, 'type': 'stats'}:
                    # Item is an evaluation
                    evaluate.append(Evaluation.model_validate(item))
                    pass
                
                case {'amount': _, 'subject': __, 'predicate': ___, 'object': ____}:
                    # Item is a histogram
                    histogram.append(HistogramSingleResult.model_validate(item))
                    pass
                
                case _: 
                    log().debug(f"Unknown item: {str(item)}")
                    print("Unknown item")
                    pass
                
        self.ruleset = Ruleset(rules = rules)
        self.predictionTasks = predictionTasks
        self.evaluate = evaluate
        self.histogram = HistogramResult(list = histogram)
    
    def get_ruleset(self) -> Ruleset:
        """Returns the ruleset generated by RDFRules.
        """
        return self.ruleset
    
    def get_evaluations(self) -> List[Evaluation]:
        """Returns the list of evaluations.
        """
        return self.evaluate
    
    def get_histogram(self) -> HistogramResult:
        """Returns the histogram.
        """
        return self.histogram