"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StatefulSetUpdateStrategy = exports.StatefulSet = exports.PodManagementPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const workload = require("./workload");
/**
 * Controls how pods are created during initial scale up, when replacing pods on nodes,
 * or when scaling down.
 *
 * The default policy is `OrderedReady`, where pods are created in increasing order
 * (pod-0, then pod-1, etc) and the controller will wait until each pod is ready before
 * continuing. When scaling down, the pods are removed in the opposite order.
 *
 * The alternative policy is `Parallel` which will create pods in parallel to match the
 * desired scale without waiting, and on scale down will delete all pods at once.
 */
var PodManagementPolicy;
(function (PodManagementPolicy) {
    PodManagementPolicy["ORDERED_READY"] = "OrderedReady";
    PodManagementPolicy["PARALLEL"] = "Parallel";
})(PodManagementPolicy = exports.PodManagementPolicy || (exports.PodManagementPolicy = {}));
/**
 * StatefulSet is the workload API object used to manage stateful applications.
 *
 * Manages the deployment and scaling of a set of Pods, and provides guarantees
 * about the ordering and uniqueness of these Pods.
 *
 * Like a Deployment, a StatefulSet manages Pods that are based on an identical
 * container spec. Unlike a Deployment, a StatefulSet maintains a sticky identity
 * for each of their Pods. These pods are created from the same spec, but are not
 * interchangeable: each has a persistent identifier that it maintains across any
 * rescheduling.
 *
 * If you want to use storage volumes to provide persistence for your workload, you
 * can use a StatefulSet as part of the solution. Although individual Pods in a StatefulSet
 * are susceptible to failure, the persistent Pod identifiers make it easier to match existing
 * volumes to the new Pods that replace any that have failed.
 *
 * Using StatefulSets
 * ------------------
 * StatefulSets are valuable for applications that require one or more of the following.
 *
 * - Stable, unique network identifiers.
 * - Stable, persistent storage.
 * - Ordered, graceful deployment and scaling.
 * - Ordered, automated rolling updates.
 */
class StatefulSet extends workload.Workload {
    constructor(scope, id, props) {
        var _c, _d, _e, _f;
        super(scope, id, props);
        this.resourceType = 'statefulsets';
        this.apiObject = new k8s.KubeStatefulSet(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this._service = props.service;
        this.apiObject.addDependency(this._service);
        this.replicas = (_c = props.replicas) !== null && _c !== void 0 ? _c : 1;
        this.strategy = (_d = props.strategy) !== null && _d !== void 0 ? _d : StatefulSetUpdateStrategy.rollingUpdate(),
            this.podManagementPolicy = (_e = props.podManagementPolicy) !== null && _e !== void 0 ? _e : PodManagementPolicy.ORDERED_READY;
        this.minReady = (_f = props.minReady) !== null && _f !== void 0 ? _f : cdk8s_1.Duration.seconds(0);
        const selectors = Object.entries(this.matchLabels);
        for (const [k, v] of selectors) {
            this._service.addSelector(k, v);
        }
    }
    /**
      * @internal
      */
    _toKube() {
        return {
            serviceName: this._service.name,
            replicas: this.replicas,
            minReadySeconds: this.minReady.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: this._toLabelSelector(),
            podManagementPolicy: this.podManagementPolicy,
            updateStrategy: this.strategy._toKube(),
        };
    }
}
exports.StatefulSet = StatefulSet;
_a = JSII_RTTI_SYMBOL_1;
StatefulSet[_a] = { fqn: "cdk8s-plus-22.StatefulSet", version: "2.0.0-beta.15" };
/**
 * StatefulSet update strategies.
 */
class StatefulSetUpdateStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * The controller will not automatically update the Pods in a StatefulSet.
     * Users must manually delete Pods to cause the controller to create new Pods
     * that reflect modifications.
     */
    static onDelete() {
        return new StatefulSetUpdateStrategy({
            type: 'OnDelete',
        });
    }
    /**
     * The controller will delete and recreate each Pod in the StatefulSet.
     * It will proceed in the same order as Pod termination (from the largest ordinal to the smallest),
     * updating each Pod one at a time. The Kubernetes control plane waits until an updated
     * Pod is Running and Ready prior to updating its predecessor.
     */
    static rollingUpdate(options = {}) {
        var _c;
        return new StatefulSetUpdateStrategy({
            type: 'RollingUpdate',
            rollingUpdate: { partition: (_c = options.partition) !== null && _c !== void 0 ? _c : 0 },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.StatefulSetUpdateStrategy = StatefulSetUpdateStrategy;
_b = JSII_RTTI_SYMBOL_1;
StatefulSetUpdateStrategy[_b] = { fqn: "cdk8s-plus-22.StatefulSetUpdateStrategy", version: "2.0.0-beta.15" };
//# sourceMappingURL=data:application/json;base64,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