"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IngressBackend = exports.Ingress = exports.HttpIngressPathType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const k8s = require("./imports/k8s");
/**
 * Specify how the path is matched against request paths.
 *
 * @see https://kubernetes.io/docs/concepts/services-networking/ingress/#path-types
 */
var HttpIngressPathType;
(function (HttpIngressPathType) {
    /**
     * Matches the URL path exactly.
     */
    HttpIngressPathType["PREFIX"] = "Prefix";
    /**
     * Matches based on a URL path prefix split by '/'.
     */
    HttpIngressPathType["EXACT"] = "Exact";
    /**
     * Matching is specified by the underlying IngressClass.
     */
    HttpIngressPathType["IMPLEMENTATION_SPECIFIC"] = "ImplementationSpecific";
})(HttpIngressPathType = exports.HttpIngressPathType || (exports.HttpIngressPathType = {}));
/**
 * Ingress is a collection of rules that allow inbound connections to reach the
 * endpoints defined by a backend. An Ingress can be configured to give services
 * externally-reachable urls, load balance traffic, terminate SSL, offer name
 * based virtual hosting etc.
 */
class Ingress extends base.Resource {
    constructor(scope, id, props = {}) {
        var _c;
        super(scope, id);
        this.resourceType = 'ingresses';
        this._rulesPerHost = {};
        this._tlsConfig = [];
        this.apiObject = new k8s.KubeIngress(this, 'Resource', {
            metadata: props.metadata,
            spec: {
                defaultBackend: cdk8s_1.Lazy.any({ produce: () => { var _c; return (_c = this._defaultBackend) === null || _c === void 0 ? void 0 : _c._toKube(); } }),
                rules: cdk8s_1.Lazy.any({ produce: () => this.synthRules() }),
                tls: cdk8s_1.Lazy.any({ produce: () => this.tlsConfig() }),
            },
        });
        if (props.defaultBackend) {
            this.addDefaultBackend(props.defaultBackend);
        }
        this.addRules(...(_c = props.rules) !== null && _c !== void 0 ? _c : []);
        if (props.tls) {
            this.addTls(props.tls);
        }
        this.node.addValidation({ validate: () => this._validate() });
    }
    _validate() {
        if (!this._defaultBackend && Object.keys(this._rulesPerHost).length === 0) {
            return ['ingress with no rules or default backend'];
        }
        return [];
    }
    /**
     * Defines the default backend for this ingress. A default backend capable of
     * servicing requests that don't match any rule.
     *
     * @param backend The backend to use for requests that do not match any rule.
     */
    addDefaultBackend(backend) {
        this.addRules({ backend });
    }
    /**
     * Specify a default backend for a specific host name. This backend will be used as a catch-all for requests
     * targeted to this host name (the `Host` header matches this value).
     *
     * @param host The host name to match
     * @param backend The backend to route to
     */
    addHostDefaultBackend(host, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend });
    }
    /**
     * Adds an ingress rule applied to requests to a specific host and a specific
     * HTTP path (the `Host` header matches this value).
     *
     * @param host The host name
     * @param path The HTTP path
     * @param backend The backend to route requests to
     * @param pathType How the path is matched against request paths
     */
    addHostRule(host, path, backend, pathType) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend, path, pathType });
    }
    /**
     * Adds an ingress rule applied to requests sent to a specific HTTP path.
     *
     * @param path The HTTP path
     * @param backend The backend to route requests to
     * @param pathType How the path is matched against request paths
     */
    addRule(path, backend, pathType) {
        this.addRules({ backend, path, pathType });
    }
    /**
     * Adds rules to this ingress.
     * @param rules The rules to add
     */
    addRules(...rules) {
        var _c, _d, _e, _f;
        for (const rule of rules) {
            // default backend is not really a rule
            if (!rule.host && !rule.path) {
                if (this._defaultBackend) {
                    throw new Error('a default backend is already defined for this ingress');
                }
                this._defaultBackend = rule.backend;
                continue;
            }
            const host = (_c = rule.host) !== null && _c !== void 0 ? _c : '';
            const backend = rule.backend;
            const path = (_d = rule.path) !== null && _d !== void 0 ? _d : '/';
            const pathType = (_e = rule.pathType) !== null && _e !== void 0 ? _e : HttpIngressPathType.PREFIX;
            if (path && !path.startsWith('/')) {
                throw new Error(`ingress paths must begin with a "/": ${path}`);
            }
            const routes = this._rulesPerHost[host] = (_f = this._rulesPerHost[host]) !== null && _f !== void 0 ? _f : [];
            // check if we already have a rule for this host/path
            if (routes.find(r => r.path === path)) {
                throw new Error(`there is already an ingress rule for ${host}${path}`);
            }
            routes.push({
                backend: backend._toKube(),
                path,
                pathType,
            });
        }
    }
    synthRules() {
        const rules = new Array();
        for (const [host, paths] of Object.entries(this._rulesPerHost)) {
            rules.push({
                host: host ? host : undefined,
                http: { paths: paths.sort(sortByPath) },
            });
        }
        return rules.length > 0 ? rules : undefined;
    }
    addTls(tls) {
        this._tlsConfig.push(...tls);
    }
    tlsConfig() {
        var _c;
        if (this._tlsConfig.length == 0) {
            return undefined;
        }
        const tls = new Array();
        for (const entry of this._tlsConfig) {
            tls.push({
                hosts: entry.hosts,
                secretName: (_c = entry.secret) === null || _c === void 0 ? void 0 : _c.name,
            });
        }
        return tls;
    }
}
exports.Ingress = Ingress;
_a = JSII_RTTI_SYMBOL_1;
Ingress[_a] = { fqn: "cdk8s-plus-22.Ingress", version: "2.0.0-beta.15" };
/**
 * The backend for an ingress path.
 */
class IngressBackend {
    constructor(backend) {
        this.backend = backend;
    }
    /**
     * A Kubernetes `Service` to use as the backend for this path.
     * @param serv The service object.
     */
    static fromService(serv, options = {}) {
        if (serv.ports.length === 0) {
            throw new Error('service does not expose any ports');
        }
        let servicePort;
        if (serv.ports.length === 1) {
            servicePort = serv.ports[0].port;
        }
        else {
            if (options.port !== undefined) {
                const found = serv.ports.find(p => p.port === options.port);
                if (found) {
                    servicePort = found.port;
                }
                else {
                    throw new Error(`service exposes ports ${serv.ports.map(p => p.port).join(',')} but backend is defined to use port ${options.port}`);
                }
            }
            else {
                throw new Error(`unable to determine service port since service exposes multiple ports: ${serv.ports.map(x => x.port).join(',')}`);
            }
        }
        if (options.port !== undefined && servicePort !== options.port) {
            throw new Error(`backend defines port ${options.port} but service exposes port ${servicePort}`);
        }
        return new IngressBackend({
            service: {
                name: serv.name,
                port: { number: servicePort },
            },
        });
    }
    /**
     * A Resource backend is an ObjectRef to another Kubernetes resource
     * within the same namespace as the Ingress object.
     * A common usage for a Resource backend is to ingress data to an object
     * storage backend with static assets.
     */
    static fromResource(resource) {
        return new IngressBackend({
            resource: {
                kind: resource.kind,
                name: resource.name,
                apiGroup: resource.apiGroup,
            },
        });
    }
    /**
     * @internal
     */
    _toKube() { return this.backend; }
}
exports.IngressBackend = IngressBackend;
_b = JSII_RTTI_SYMBOL_1;
IngressBackend[_b] = { fqn: "cdk8s-plus-22.IngressBackend", version: "2.0.0-beta.15" };
function sortByPath(lhs, rhs) {
    var _c, _d;
    const p1 = (_c = lhs.path) !== null && _c !== void 0 ? _c : '';
    const p2 = (_d = rhs.path) !== null && _d !== void 0 ? _d : '';
    return p1.localeCompare(p2);
}
//# sourceMappingURL=data:application/json;base64,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