"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentStrategy = exports.PercentOrAbsolute = exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const service = require("./service");
const workload = require("./workload");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
* ---------
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends workload.Workload {
    constructor(scope, id, props = {}) {
        var _d, _e, _f, _g;
        super(scope, id, props);
        this.resourceType = 'deployments';
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.minReady = (_d = props.minReady) !== null && _d !== void 0 ? _d : cdk8s_1.Duration.seconds(0);
        this.progressDeadline = (_e = props.progressDeadline) !== null && _e !== void 0 ? _e : cdk8s_1.Duration.seconds(600);
        if (this.progressDeadline.toSeconds() <= this.minReady.toSeconds()) {
            throw new Error(`'progressDeadline' (${this.progressDeadline.toSeconds()}s) must be greater than 'minReady' (${this.minReady.toSeconds()}s)`);
        }
        this.replicas = (_f = props.replicas) !== null && _f !== void 0 ? _f : 1;
        this.strategy = (_g = props.strategy) !== null && _g !== void 0 ? _g : DeploymentStrategy.rollingUpdate();
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options = {}) {
        var _d;
        const ser = new service.Service(this, 'Service', {
            metadata: options.name ? { name: options.name } : undefined,
            type: (_d = options.serviceType) !== null && _d !== void 0 ? _d : service.ServiceType.CLUSTER_IP,
        });
        ser.addDeployment(this, { protocol: options.protocol, targetPort: options.targetPort, port: options.port });
        return ser;
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path, options = {}) {
        const ser = this.exposeViaService(options);
        return ser.exposeViaIngress(path, options);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.replicas,
            minReadySeconds: this.minReady.toSeconds(),
            progressDeadlineSeconds: this.progressDeadline.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: this._toLabelSelector(),
            strategy: this.strategy._toKube(),
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-22.Deployment", version: "2.0.0-beta.15" };
/**
 * Union like class repsenting either a ration in
 * percents or an absolute number.
 */
class PercentOrAbsolute {
    constructor(value) {
        this.value = value;
    }
    /**
     * Percent ratio.
     */
    static percent(percent) {
        return new PercentOrAbsolute(`${percent}%`);
    }
    /**
     * Absolute number.
     */
    static absolute(num) {
        return new PercentOrAbsolute(num);
    }
    isZero() {
        return this.value === PercentOrAbsolute.absolute(0).value || this.value === PercentOrAbsolute.percent(0).value;
    }
}
exports.PercentOrAbsolute = PercentOrAbsolute;
_b = JSII_RTTI_SYMBOL_1;
PercentOrAbsolute[_b] = { fqn: "cdk8s-plus-22.PercentOrAbsolute", version: "2.0.0-beta.15" };
/**
 * Deployment strategies.
 */
class DeploymentStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * All existing Pods are killed before new ones are created.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#recreate-deployment
     */
    static recreate() {
        return new DeploymentStrategy({
            type: 'Recreate',
        });
    }
    static rollingUpdate(options = {}) {
        var _d, _e;
        const maxSurge = (_d = options.maxSurge) !== null && _d !== void 0 ? _d : PercentOrAbsolute.percent(25);
        const maxUnavailable = (_e = options.maxUnavailable) !== null && _e !== void 0 ? _e : PercentOrAbsolute.percent(25);
        if (maxSurge.isZero() && maxUnavailable.isZero()) {
            throw new Error('\'maxSurge\' and \'maxUnavailable\' cannot be both zero');
        }
        return new DeploymentStrategy({
            type: 'RollingUpdate',
            rollingUpdate: { maxSurge, maxUnavailable },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_c = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_c] = { fqn: "cdk8s-plus-22.DeploymentStrategy", version: "2.0.0-beta.15" };
//# sourceMappingURL=data:application/json;base64,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