"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Env = exports.extractContainerPorts = exports.EnvFrom = exports.Cpu = exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = exports.ContainerSecurityContext = exports.Protocol = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const probe = require("./probe");
const utils_1 = require("./utils");
/**
 * Network protocols.
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP.
     */
    Protocol["TCP"] = "TCP";
    /**
     * UDP.
     */
    Protocol["UDP"] = "UDP";
    /**
     * SCTP.
     */
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Container security attributes and settings.
 */
class ContainerSecurityContext {
    constructor(props = {}) {
        this.ensureNonRoot = props.ensureNonRoot ?? true;
        this.privileged = props.privileged ?? false;
        this.readOnlyRootFilesystem = props.readOnlyRootFilesystem ?? true;
        this.user = props.user ?? 25000;
        this.group = props.group ?? 26000;
        this.allowPrivilegeEscalation = props.allowPrivilegeEscalation ?? false;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            runAsNonRoot: this.ensureNonRoot,
            privileged: this.privileged,
            readOnlyRootFilesystem: this.readOnlyRootFilesystem,
            allowPrivilegeEscalation: this.allowPrivilegeEscalation,
        };
    }
}
exports.ContainerSecurityContext = ContainerSecurityContext;
_a = JSII_RTTI_SYMBOL_1;
ContainerSecurityContext[_a] = { fqn: "cdk8s-plus-22.ContainerSecurityContext", version: "2.0.0-rc.150" };
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_b = JSII_RTTI_SYMBOL_1;
EnvValue[_b] = { fqn: "cdk8s-plus-22.EnvValue", version: "2.0.0-rc.150" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        this._ports = [];
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        const defaultResourceSpec = {
            cpu: {
                request: Cpu.millis(1000),
                limit: Cpu.millis(1500),
            },
            memory: {
                request: cdk8s_1.Size.mebibytes(512),
                limit: cdk8s_1.Size.mebibytes(2048),
            },
        };
        if (props.port && props.portNumber) {
            throw new Error('Either \'port\' or \'portNumber\' is allowed. Use \'portNumber\' since \'port\' is deprecated');
        }
        const portNumber = props.portNumber ?? props.port;
        const defaultProbeConfiguration = probe.Probe.fromTcpSocket({ port: portNumber });
        this.name = props.name ?? 'main';
        this.image = props.image;
        this.portNumber = portNumber;
        this._command = props.command;
        this._args = props.args;
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup ?? (this.portNumber ? defaultProbeConfiguration : undefined);
        this._lifecycle = props.lifecycle;
        this.resources = props.resources ?? defaultResourceSpec;
        this.workingDir = props.workingDir;
        this.mounts = props.volumeMounts ?? [];
        this.imagePullPolicy = props.imagePullPolicy ?? ImagePullPolicy.ALWAYS;
        this.securityContext = new ContainerSecurityContext(props.securityContext);
        this.env = new Env(props.envFrom ?? [], props.envVariables ?? {});
        if (this.portNumber) {
            this.addPort({
                number: this.portNumber,
            });
        }
        for (const port of props.ports ?? []) {
            this.addPort(port);
        }
    }
    /**
     * @deprecated - use `portNumber`.
     */
    get port() {
        return this.portNumber;
    }
    /**
     * Ports exposed by this containers.
     * Returns a copy, use `addPort` to modify.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param storage - The storage to mount.
     */
    mount(path, storage, options = {}) {
        this.mounts.push({ path, volume: storage.asVolume(), ...options });
    }
    /**
     * Add a port to expose from this container.
     */
    addPort(port) {
        const names = this._ports.map(p => p.name).filter(x => x);
        const numbers = this._ports.map(p => p.number);
        if (port.name && names.includes(port.name)) {
            throw new Error(`Port with name ${port.name} already exists`);
        }
        if (numbers.includes(port.number)) {
            throw new Error(`Port with number ${port.number} already exists`);
        }
        this._ports.push(port);
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        for (const port of this.ports) {
            ports.push({
                containerPort: port.number,
                protocol: port.protocol?.toString(),
                name: port.name,
                hostPort: port.hostPort,
                hostIp: port.hostIp,
            });
        }
        // Resource requests and limits
        const cpuLimit = this.resources?.cpu?.limit?.amount;
        const cpuRequest = this.resources?.cpu?.request?.amount;
        const memoryLimit = this.resources?.memory?.limit;
        const memoryRequest = this.resources?.memory?.request;
        const ephemeralStorageLimit = this.resources?.ephemeralStorage?.limit;
        const ephemeralStorageRequest = this.resources?.ephemeralStorage?.request;
        const limits = {};
        const requests = {};
        if (cpuLimit) {
            limits.cpu = k8s.Quantity.fromString(cpuLimit);
        }
        if (memoryLimit) {
            limits.memory = k8s.Quantity.fromString(memoryLimit.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageLimit) {
            limits['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageLimit.toGibibytes().toString() + 'Gi');
        }
        if (cpuRequest) {
            requests.cpu = k8s.Quantity.fromString(cpuRequest);
        }
        if (memoryRequest) {
            requests.memory = k8s.Quantity.fromString(memoryRequest.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageRequest) {
            requests['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageRequest.toGibibytes().toString() + 'Gi');
        }
        let resourceRequirements = undefined;
        if (Object.keys(limits).length > 0 || Object.keys(requests).length > 0) {
            resourceRequirements = {
                limits: utils_1.undefinedIfEmpty(limits),
                requests: utils_1.undefinedIfEmpty(requests),
            };
        }
        const env = this.env._toKube();
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this.imagePullPolicy,
            ports: utils_1.undefinedIfEmpty(ports),
            volumeMounts: utils_1.undefinedIfEmpty(volumeMounts),
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: env.variables,
            envFrom: env.from,
            readinessProbe: this._readiness?._toKube(this),
            livenessProbe: this._liveness?._toKube(this),
            startupProbe: this._startup?._toKube(this),
            lifecycle: this._lifecycle ? {
                postStart: this._lifecycle.postStart?._toKube(this),
                preStop: this._lifecycle.preStop?._toKube(this),
            } : undefined,
            resources: resourceRequirements,
            securityContext: this.securityContext._toKube(),
        };
    }
}
exports.Container = Container;
_c = JSII_RTTI_SYMBOL_1;
Container[_c] = { fqn: "cdk8s-plus-22.Container", version: "2.0.0-rc.150" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
/**
 * Represents the amount of CPU.
 * The amount can be passed as millis or units.
 */
class Cpu {
    constructor(amount) {
        this.amount = amount;
    }
    static millis(amount) {
        return new Cpu(amount + 'm');
    }
    static units(amount) {
        return new Cpu(amount.toString());
    }
}
exports.Cpu = Cpu;
_d = JSII_RTTI_SYMBOL_1;
Cpu[_d] = { fqn: "cdk8s-plus-22.Cpu", version: "2.0.0-rc.150" };
/**
 * A collection of env variables defined in other resources.
 */
class EnvFrom {
    constructor(configMap, prefix, sec) {
        this.configMap = configMap;
        this.prefix = prefix;
        this.sec = sec;
    }
    ;
    /**
     * @internal
     */
    _toKube() {
        return {
            configMapRef: this.configMap ? {
                name: this.configMap.name,
            } : undefined,
            secretRef: this.sec ? {
                name: this.sec.name,
            } : undefined,
            prefix: this.prefix,
        };
    }
}
exports.EnvFrom = EnvFrom;
_e = JSII_RTTI_SYMBOL_1;
EnvFrom[_e] = { fqn: "cdk8s-plus-22.EnvFrom", version: "2.0.0-rc.150" };
function extractContainerPorts(selector) {
    if (!selector) {
        return [];
    }
    // we don't use instanceof intentionally since it can create
    // cyclic import problems.
    const containers = selector.containers ?? [];
    return containers.flatMap(c => c.ports);
}
exports.extractContainerPorts = extractContainerPorts;
/**
 * Container environment variables.
 */
class Env {
    constructor(sources, variables) {
        this._sources = sources;
        this._variables = variables;
    }
    /**
     * Selects a ConfigMap to populate the environment variables with.
     * The contents of the target ConfigMap's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromConfigMap(configMap, prefix) {
        return new EnvFrom(configMap, prefix, undefined);
    }
    /**
     * Selects a Secret to populate the environment variables with.
     * The contents of the target Secret's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromSecret(secr) {
        return new EnvFrom(undefined, undefined, secr);
    }
    /**
     * Add a single variable by name and value.
     * The variable value can come from various dynamic sources such a secrets of config maps.
     * Use `EnvValue.fromXXX` to select sources.
     */
    addVariable(name, value) {
        this._variables[name] = value;
    }
    /**
     * The environment variables for this container.
     * Returns a copy. To add environment variables use `container.env.addVariable()`.
     */
    get variables() {
        return { ...this._variables };
    }
    /**
     * Add a collection of variables by copying from another source.
     * Use `Env.fromXXX` functions to select sources.
     */
    copyFrom(from) {
        this._sources.push(from);
    }
    /**
     * The list of sources used to populate the container environment,
     * in addition to the `variables`.
     *
     * Returns a copy. To add a source use `container.env.copyFrom()`.
     */
    get sources() {
        return [...this._sources];
    }
    renderEnv(env) {
        const result = new Array();
        for (const [name, v] of Object.entries(env)) {
            result.push({
                name,
                value: v.value,
                valueFrom: v.valueFrom,
            });
        }
        return result;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            from: utils_1.undefinedIfEmpty(this._sources.map(s => s._toKube())),
            variables: utils_1.undefinedIfEmpty(this.renderEnv(this._variables)),
        };
    }
}
exports.Env = Env;
_f = JSII_RTTI_SYMBOL_1;
Env[_f] = { fqn: "cdk8s-plus-22.Env", version: "2.0.0-rc.150" };
//# sourceMappingURL=data:application/json;base64,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