"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerPort = exports.LoadBalancer = exports.LoadBalancingProtocol = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * A load balancer with a single listener
 *
 * Routes to a fleet of of instances in a VPC.
 */
class LoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        var _c;
        super(scope, id);
        /**
         * An object controlling specifically the connections for each listener added to this load balancer
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancing_LoadBalancerProps(props);
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.any({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: (_c = props.crossZone) !== null && _c !== void 0 ? _c : true,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        if (props.accessLoggingPolicy !== undefined) {
            this.elb.accessLoggingPolicy = props.accessLoggingPolicy;
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * Add a backend to the load balancer
     *
     * @returns A ListenerPort object that controls connections to the listener port
     */
    addListener(listener) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancing_LoadBalancerListener(listener);
        if (listener.sslCertificateArn && listener.sslCertificateId) {
            throw new Error('"sslCertificateId" is deprecated, please use "sslCertificateArn" only.');
        }
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const sslCertificateArn = listener.sslCertificateArn || listener.sslCertificateId;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: sslCertificateArn,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    addTarget(target) {
        jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancing_ILoadBalancerTarget(target);
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @attribute
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @attribute
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
_a = JSII_RTTI_SYMBOL_1;
LoadBalancer[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancing.LoadBalancer", version: "1.152.0" };
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *    const listener = elb.addListener(...);
 *
 *    listener.connections.allowDefaultPortFromAnyIPv4();
 *    // or
 *    instance.connections.allowToDefaultPort(listener);
 */
class ListenerPort {
    constructor(securityGroup, defaultPort) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
exports.ListenerPort = ListenerPort;
_b = JSII_RTTI_SYMBOL_1;
ListenerPort[_b] = { fqn: "@aws-cdk/aws-elasticloadbalancing.ListenerPort", version: "1.152.0" };
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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