"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsiteOrigin = exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_origins_1 = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const cloudfront_web_acl_1 = require("./cloudfront-web-acl");
const DEFAULT_RUNTIME_CONFIG_FILENAME = "runtime-config.json";
/**
 * Deploys a Static Website using by default a private S3 bucket as an origin and Cloudfront as the entrypoint.
 *
 * This construct configures a webAcl containing rules that are generally applicable to web applications. This
 * provides protection against exploitation of a wide range of vulnerabilities, including some of the high risk
 * and commonly occurring vulnerabilities described in OWASP publications such as OWASP Top 10.
 *
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.validateProps = (props) => {
            this.validateEncryptionSettings(props);
            props.runtimeOptions && this.validateRuntimeConfig(props.runtimeOptions);
            props.websiteBucket && this.validateBucketConfig(props.websiteBucket);
        };
        this.validateRuntimeConfig = (config) => {
            if (!config) {
                throw new Error("validateRuntimeConfig only accepts non-null RuntimeOptions.");
            }
            if (config.jsonFileName && !config.jsonFileName.endsWith(".json")) {
                throw new Error("RuntimeOptions.jsonFileName must be a json file.");
            }
        };
        this.validateBucketConfig = (bucket) => {
            if (bucket.isWebsite) {
                throw new Error("Website buckets cannot be configured as websites as this will break Cloudfront hosting!");
            }
        };
        this.validateEncryptionSettings = ({ defaultWebsiteBucketEncryption, defaultWebsiteBucketEncryptionKey, }) => {
            if (defaultWebsiteBucketEncryptionKey &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS) {
                throw new Error("Bucket encryption should be set to KMS if providing a defaultWebsiteBucketEncryptionKey.");
            }
            if (defaultWebsiteBucketEncryption &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.KMS &&
                defaultWebsiteBucketEncryption !== aws_s3_1.BucketEncryption.S3_MANAGED) {
                throw new Error("Only KMS and S3_MANAGED encryption are supported on the default bucket.");
            }
        };
        this.suppressCDKNagViolations = (props) => {
            const stack = aws_cdk_lib_1.Stack.of(this);
            !props.distributionProps?.certificate &&
                cdk_nag_1.NagSuppressions.addResourceSuppressions(this.cloudFrontDistribution, [
                    {
                        id: "AwsSolutions-CFR4",
                        reason: "Certificate is not mandatory therefore the Cloudfront certificate will be used.",
                    },
                ]);
            cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                {
                    id: "AwsSolutions-L1",
                    reason: "Latest runtime cannot be configured. CDK will need to upgrade the BucketDeployment construct accordingly.",
                },
            ]);
            cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                {
                    id: "AwsSolutions-IAM5",
                    reason: "All Policies have been scoped to a Bucket. Given Buckets can contain arbitrary content, wildcard resources with bucket scope are required.",
                    appliesTo: [
                        {
                            regex: "/^Action::s3:.*$/g",
                        },
                        {
                            regex: `/^Resource::.*$/g`,
                        },
                    ],
                },
            ]);
            cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                {
                    id: "AwsSolutions-IAM4",
                    reason: "Buckets can contain arbitrary content, therefore wildcard resources under a bucket are required.",
                    appliesTo: [
                        {
                            regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole$/g`,
                        },
                    ],
                },
            ]);
        };
        this.validateProps(props);
        // S3 Bucket to hold website files
        this.websiteBucket =
            props.websiteBucket ??
                new aws_s3_1.Bucket(this, "WebsiteBucket", {
                    versioned: true,
                    enforceSSL: true,
                    autoDeleteObjects: true,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                    encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                    publicReadAccess: false,
                    blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                    serverAccessLogsPrefix: "access-logs",
                });
        // Web ACL
        const { distributionProps } = props;
        const webAclArn = distributionProps?.webAclId ??
            new cloudfront_web_acl_1.CloudfrontWebAcl(this, "WebsiteAcl", props.webAclProps).webAclArn;
        // Cloudfront Distribution
        const logBucket = props.distributionProps?.logBucket ||
            new aws_s3_1.Bucket(this, "DistributionLogBucket", {
                enforceSSL: true,
                autoDeleteObjects: true,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                encryption: props.defaultWebsiteBucketEncryption ?? aws_s3_1.BucketEncryption.S3_MANAGED,
                encryptionKey: props.defaultWebsiteBucketEncryptionKey,
                publicReadAccess: false,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                serverAccessLogsPrefix: "access-logs",
            });
        const originAccessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, "OriginAccessIdentity");
        this.websiteBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [this.websiteBucket.bucketArn],
            actions: ["s3:ListBucket"],
            principals: [originAccessIdentity.grantPrincipal],
        }));
        const defaultRootObject = distributionProps?.defaultRootObject ?? "index.html";
        this.cloudFrontDistribution = new aws_cloudfront_1.Distribution(this, "CloudfrontDistribution", {
            ...distributionProps,
            webAclId: webAclArn,
            enableLogging: true,
            logBucket: logBucket,
            defaultBehavior: {
                ...distributionProps?.defaultBehavior,
                origin: new aws_cloudfront_origins_1.S3Origin(this.websiteBucket, {
                    originAccessIdentity,
                }),
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            defaultRootObject,
            errorResponses: distributionProps?.errorResponses ?? [
                {
                    httpStatus: 404,
                    responseHttpStatus: 200,
                    responsePagePath: `/${defaultRootObject}`,
                },
            ],
        });
        // Deploy Website
        this.bucketDeployment = new aws_s3_deployment_1.BucketDeployment(this, "WebsiteDeployment", {
            sources: [
                aws_s3_deployment_1.Source.asset(props.websiteContentPath),
                ...(props.runtimeOptions
                    ? [
                        aws_s3_deployment_1.Source.jsonData(props.runtimeOptions?.jsonFileName ||
                            DEFAULT_RUNTIME_CONFIG_FILENAME, props.runtimeOptions?.jsonPayload),
                    ]
                    : []),
            ],
            destinationBucket: this.websiteBucket,
            // Files in the distribution's edge caches will be invalidated after files are uploaded to the destination bucket.
            distribution: this.cloudFrontDistribution,
        });
        this.suppressCDKNagViolations(props);
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsite", version: "0.12.18" };
/**
 * If passing in distributionProps, the default behaviour.origin is a required parameter. An instance of this class can be passed in
 * to make the compiler happy.
 */
class StaticWebsiteOrigin {
    bind(_scope, _options) {
        throw new Error("This should never be called");
    }
}
exports.StaticWebsiteOrigin = StaticWebsiteOrigin;
_b = JSII_RTTI_SYMBOL_1;
StaticWebsiteOrigin[_b] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsiteOrigin", version: "0.12.18" };
//# sourceMappingURL=data:application/json;base64,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