"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudfrontWebAcl = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
/**
 * This construct creates a WAFv2 Web ACL for cloudfront in the us-east-1 region (required for cloudfront) no matter the
 * region of the parent cdk stack.
 */
class CloudfrontWebAcl extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const stack = aws_cdk_lib_1.Stack.of(this);
        const aclName = `${stack.stackName}-${id}`; // Unique per stack
        const onEventHandler = this.createOnEventHandler(stack, aclName);
        const customResource = this.createAclCustomResource(stack, aclName, onEventHandler, props);
        this.webAclId = customResource.getAttString("WebAclId");
        this.webAclArn = customResource.getAttString("WebAclArn");
    }
    /**
     * Creates an event handler for managing an ACL in us-east-1.
     *
     * @param stack containing Stack instance.
     * @param aclName name of the ACL to manage.
     * @private
     */
    createOnEventHandler(stack, aclName) {
        const onEventHandlerName = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")}OnEventHandler`;
        const onEventHandlerRole = new aws_iam_1.Role(this, "OnEventHandlerRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${onEventHandlerName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${onEventHandlerName}:*`,
                            ],
                        }),
                    ],
                }),
                wafv2: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "wafv2:CreateWebACL",
                                "wafv2:DeleteWebACL",
                                "wafv2:UpdateWebACL",
                                "wafv2:GetWebACL",
                            ],
                            resources: [
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/ipset/${aclName}-IPSet/*`,
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/webacl/${aclName}/*`,
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/managedruleset/*/*`,
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "wafv2:CreateIPSet",
                                "wafv2:DeleteIPSet",
                                "wafv2:UpdateIPSet",
                                "wafv2:GetIPSet",
                            ],
                            resources: [
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/ipset/${aclName}-IPSet/*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const onEventHandler = new aws_lambda_1.Function(this, "CloudfrontWebAclOnEventHandler", {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../lib/webacl_event_handler")),
            role: onEventHandlerRole,
            functionName: onEventHandlerName,
            handler: "index.onEvent",
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            timeout: aws_cdk_lib_1.Duration.seconds(300),
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(onEventHandlerRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "WafV2 resources have been scoped down to the ACL/IPSet level, however * is still needed as resource id's are created just in time.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:aws:wafv2:us-east-1:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:global/(.*)$/g`,
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${onEventHandlerName}:\*/g`,
                    },
                ],
            },
        ], true);
        return onEventHandler;
    }
    /**
     * Creates a Custom resource to manage the deployment of the ACL.
     *
     * @param stack containing Stack instance.
     * @param aclName name of the ACL to manage.
     * @param onEventHandler event handler to use for deployment.
     * @param props user provided properties for configuring the ACL.
     * @private
     */
    createAclCustomResource(stack, aclName, onEventHandler, props) {
        const providerFunctionName = `${onEventHandler.functionName}-Provider`;
        const providerRole = new aws_iam_1.Role(this, "CloudfrontWebAclProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "CloudfrontWebAclProvider", {
            onEventHandler,
            role: providerRole,
            providerFunctionName,
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
            {
                id: "AwsSolutions-L1",
                reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
            },
        ], true);
        return new aws_cdk_lib_1.CustomResource(this, "CFWebAclCustomResource", {
            serviceToken: provider.serviceToken,
            properties: {
                ID: aclName,
                MANAGED_RULES: props?.managedRules ?? [
                    { vendor: "AWS", name: "AWSManagedRulesCommonRuleSet" },
                ],
                CIDR_ALLOW_LIST: props?.cidrAllowList,
            },
        });
    }
}
exports.CloudfrontWebAcl = CloudfrontWebAcl;
_a = JSII_RTTI_SYMBOL_1;
CloudfrontWebAcl[_a] = { fqn: "@aws-prototyping-sdk/static-website.CloudfrontWebAcl", version: "0.12.18" };
//# sourceMappingURL=data:application/json;base64,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