import { ApiObject, ApiObjectMetadataDefinition } from 'cdk8s';
import { Construct } from 'constructs';
import { Resource, ResourceProps } from './base';
import { Container, ContainerProps } from './container';
import * as k8s from './imports/k8s';
import { RestartPolicy, IPodTemplate, PodTemplateProps } from './pod';
import { Service } from './service';
import { IServiceAccount } from './service-account';
import { Volume } from './volume';
/**
 * Controls how pods are created during initial scale up, when replacing pods on nodes, or when scaling down.
 *
 * The default policy is `OrderedReady`, where pods are created in increasing order
 * (pod-0, then pod-1, etc) and the controller will wait until each pod is ready before
 * continuing. When scaling down, the pods are removed in the opposite order.
 *
 * The alternative policy is `Parallel` which will create pods in parallel to match the
 * desired scale without waiting, and on scale down will delete all pods at once.
 *
 * @stability stable
 */
export declare enum PodManagementPolicy {
    /**
     * @stability stable
     */
    ORDERED_READY = "OrderedReady",
    /**
     * @stability stable
     */
    PARALLEL = "Parallel"
}
/**
 * Properties for initialization of `StatefulSet`.
 *
 * @stability stable
 */
export interface StatefulSetProps extends ResourceProps, PodTemplateProps {
    /**
     * Service to associate with the statefulset.
     *
     * @stability stable
     */
    readonly service: Service;
    /**
     * Number of desired pods.
     *
     * @default 1
     * @stability stable
     */
    readonly replicas?: number;
    /**
     * Automatically allocates a pod selector for this statefulset.
     *
     * If this is set to `false` you must define your selector through
     * `statefulset.podMetadata.addLabel()` and `statefulset.selectByLabel()`.
     *
     * @default true
     * @stability stable
     */
    readonly defaultSelector?: boolean;
    /**
     * Pod management policy to use for this statefulset.
     *
     * @default PodManagementPolicy.ORDERED_READY
     * @stability stable
     */
    readonly podManagementPolicy?: PodManagementPolicy;
}
/**
 * StatefulSet is the workload API object used to manage stateful applications.
 *
 * Manages the deployment and scaling of a set of Pods, and provides guarantees
 * about the ordering and uniqueness of these Pods.
 *
 * Like a Deployment, a StatefulSet manages Pods that are based on an identical
 * container spec. Unlike a Deployment, a StatefulSet maintains a sticky identity
 * for each of their Pods. These pods are created from the same spec, but are not
 * interchangeable: each has a persistent identifier that it maintains across any
 * rescheduling.
 *
 * If you want to use storage volumes to provide persistence for your workload, you
 * can use a StatefulSet as part of the solution. Although individual Pods in a StatefulSet
 * are susceptible to failure, the persistent Pod identifiers make it easier to match existing
 * volumes to the new Pods that replace any that have failed.
 *
 * Using StatefulSets
 * ------------------
 * StatefulSets are valuable for applications that require one or more of the following.
 *
 * - Stable, unique network identifiers.
 * - Stable, persistent storage.
 * - Ordered, graceful deployment and scaling.
 * - Ordered, automated rolling updates.
 *
 * @stability stable
 */
export declare class StatefulSet extends Resource implements IPodTemplate {
    /**
     * Number of desired pods.
     *
     * @stability stable
     */
    readonly replicas: number;
    /**
     * Management policy to use for the set.
     *
     * @stability stable
     */
    readonly podManagementPolicy: PodManagementPolicy;
    /**
     * The underlying cdk8s API object.
     *
     * @see base.Resource.apiObject
     * @stability stable
     */
    protected readonly apiObject: ApiObject;
    private readonly _podTemplate;
    private readonly _labelSelector;
    private readonly _service;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: StatefulSetProps);
    /**
     * Provides read/write access to the underlying pod metadata of the resource.
     *
     * @stability stable
     */
    get podMetadata(): ApiObjectMetadataDefinition;
    /**
     * The labels this statefulset will match against in order to select pods.
     *
     * Returns a a copy. Use `selectByLabel()` to add labels.
     *
     * @stability stable
     */
    get labelSelector(): Record<string, string>;
    /**
     * The containers belonging to the pod.
     *
     * Use `addContainer` to add containers.
     *
     * @stability stable
     */
    get containers(): Container[];
    /**
     * The volumes associated with this pod.
     *
     * Use `addVolume` to add volumes.
     *
     * @stability stable
     */
    get volumes(): Volume[];
    /**
     * Restart policy for all containers within the pod.
     *
     * @stability stable
     */
    get restartPolicy(): RestartPolicy | undefined;
    /**
     * The service account used to run this pod.
     *
     * @stability stable
     */
    get serviceAccount(): IServiceAccount | undefined;
    /**
     * Configure a label selector to this deployment.
     *
     * Pods that have the label will be selected by deployments configured with this spec.
     *
     * @param key - The label key.
     * @param value - The label value.
     * @stability stable
     */
    selectByLabel(key: string, value: string): void;
    /**
     * Add a container to the pod.
     *
     * @stability stable
     */
    addContainer(container: ContainerProps): Container;
    /**
     * Add a volume to the pod.
     *
     * @stability stable
     */
    addVolume(volume: Volume): void;
    /**
      * @internal
      */
    _toKube(): k8s.StatefulSetSpec;
}
