"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const ddb = require("@aws-cdk/aws-dynamodb");
const api = require("@aws-cdk/aws-apigateway");
test("snapshot test ApiGatewayToDynamoDB default params", () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {};
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb-default", apiGatewayToDynamoDBProps);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test("check properties", () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {};
    const construct = new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb-default", apiGatewayToDynamoDBProps);
    expect(construct.dynamoTable !== null);
    expect(construct.apiGateway !== null);
    expect(construct.apiGatewayRole !== null);
    expect(construct.apiGatewayCloudWatchRole !== null);
    expect(construct.apiGatewayLogGroup !== null);
});
test("check allow CRUD operations", () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        allowReadOperation: true,
        allowCreateOperation: true,
        createRequestTemplate: "{}",
        allowDeleteOperation: true,
        allowUpdateOperation: true,
        updateRequestTemplate: "{}",
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb", apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:PutItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
                {
                    Action: "dynamodb:Query",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
                {
                    Action: "dynamodb:UpdateItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
                {
                    Action: "dynamodb:DeleteItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
            ],
            Version: "2012-10-17",
        },
        PolicyName: "testapigatewaydynamodbapigatewayroleDefaultPolicy43AC565D",
        Roles: [
            {
                Ref: "testapigatewaydynamodbapigatewayrole961B19C4",
            },
        ],
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM",
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "PUT",
        AuthorizationType: "AWS_IAM",
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "DELETE",
        AuthorizationType: "AWS_IAM",
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Resource", {
        PathPart: "{id}",
    });
});
test("check allow read and update only", () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        allowUpdateOperation: true,
        updateRequestTemplate: "{}",
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb", apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:Query",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
                {
                    Action: "dynamodb:UpdateItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
            ],
            Version: "2012-10-17",
        },
        PolicyName: "testapigatewaydynamodbapigatewayroleDefaultPolicy43AC565D",
        Roles: [
            {
                Ref: "testapigatewaydynamodbapigatewayrole961B19C4",
            },
        ],
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM",
    });
});
test("check using custom partition key for dynamodb", () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        dynamoTableProps: {
            partitionKey: {
                name: "page_id",
                type: ddb.AttributeType.STRING,
            },
        },
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb", apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::ApiGateway::Resource", {
        PathPart: "{page_id}",
    });
});
test("override apiGatewayProps for api gateway", () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        apiGatewayProps: {
            description: "This is a sample description for api gateway",
        },
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb", apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::ApiGateway::RestApi", {
        Description: "This is a sample description for api gateway",
    });
});
test("Test deployment ApiGateway AuthorizationType override", () => {
    const stack = new core_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, "api-gateway-dynamodb", {
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE,
            },
        },
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "NONE",
    });
});
test("Test deployment with existing DynamoDB table", () => {
    const oddPartitionKeyName = 'oddName';
    const oddReadCapacity = 23;
    const stack = new core_1.Stack();
    const table = new ddb.Table(stack, "existing-table", {
        partitionKey: {
            name: oddPartitionKeyName,
            type: ddb.AttributeType.STRING,
        },
        readCapacity: oddReadCapacity
    });
    const apiGatewayToDynamoDBProps = {
        existingTableObj: table
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb-default", apiGatewayToDynamoDBProps);
    // Confirm there is only the one table
    expect(stack).toCountResources("AWS::DynamoDB::Table", 1);
    // Confirm that the one table is the one create here
    expect(stack).toHaveResourceLike("AWS::DynamoDB::Table", {
        ProvisionedThroughput: {
            ReadCapacityUnits: oddReadCapacity,
        }
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Resource", {
        PathPart: `{${oddPartitionKeyName}}`,
    });
});
test("check setting allowReadOperation=false for dynamodb", () => {
    const stack1 = new core_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack1, "test-api-gateway-dynamodb1", {
        allowReadOperation: true,
        allowDeleteOperation: true
    });
    // Expect two APIG Methods (GET, DELETE) for allowReadOperation and allowDeleteOperation
    expect(stack1).toCountResources("AWS::ApiGateway::Method", 2);
    const stack2 = new core_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack2, "test-api-gateway-dynamodb2", {
        allowReadOperation: false,
        allowDeleteOperation: true
    });
    // Expect only one APIG Method (DELETE) for allowDeleteOperation
    expect(stack2).toCountResources("AWS::ApiGateway::Method", 1);
});
//# sourceMappingURL=data:application/json;base64,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