import { IDatabase } from "@aws-cdk/aws-glue-alpha";
import { CfnFlow } from "aws-cdk-lib/aws-appflow";
import { IRole } from "aws-cdk-lib/aws-iam";
import { ConnectorType } from "../core/connectors/connector-type";
import { IFlow } from "../core/flows";
import { S3Location } from "../core/s3-location";
import { IDestination } from "../core/vertices/destination";
export interface S3FileAggregation {
    readonly type?: S3OutputAggregationType;
    /**
     * The maximum size, in MB, of the file containing portion of incoming data
     */
    readonly fileSize?: number;
}
export interface S3OutputFilePrefix {
    readonly hierarchy?: S3OutputFilePrefixHierarchy[];
    readonly format?: S3OutputFilePrefixFormat;
    readonly type?: S3OutputFilePrefixType;
}
export interface S3OutputFormatting {
    /**
     * Sets an aggregation approach per flow run
     */
    readonly aggregation?: S3FileAggregation;
    /**
     * Sets the file type for the output files
     * @default - JSON file type
     */
    readonly fileType?: S3OutputFileType;
    /**
     * Sets a prefix approach for files generated during a flow execution
     */
    readonly filePrefix?: S3OutputFilePrefix;
    /**
     * Specifies whether AppFlow should attempt data type mapping from source when the destination output file type is Parquet
     * @default - do not preserve source data files
     */
    readonly preserveSourceDataTypes?: boolean;
}
export declare enum S3OutputAggregationType {
    NONE = "None",
    SINGLE_FILE = "SingleFile"
}
/**
 * Output file type supported by Amazon S3 Destination connector
 */
export declare enum S3OutputFileType {
    /**
     * CSV file type
     */
    CSV = "CSV",
    /**
     * JSON file type
     */
    JSON = "JSON",
    /**
     * Parquet file type
     */
    PARQUET = "PARQUET"
}
export declare enum S3OutputFilePrefixHierarchy {
    EXECUTION_ID = "EXECUTION_ID",
    SCHEMA_VERSION = "SCHEMA_VERSION"
}
export declare enum S3OutputFilePrefixFormat {
    DAY = "DAY",
    HOUR = "HOUR",
    MINUTE = "MINUTE",
    MONTH = "MONTH",
    YEAR = "YEAR"
}
export declare enum S3OutputFilePrefixType {
    FILENAME = "FILENAME",
    PATH = "PATH",
    PATH_AND_FILE = "PATH_AND_FILE"
}
export interface S3Catalog {
    /**
     * The IAM Role that will be used for data catalog operations
     * @default - A new role will be created
     */
    readonly role?: IRole;
    /**
     * The AWS Glue database that will contain the tables created when the flow executes
     */
    readonly database: IDatabase;
    /**
     * The prefix for the tables created in the AWS Glue database
     */
    readonly tablePrefix: string;
}
export interface S3DestinationProps {
    /**
     * The S3 location of the files with the retrieved data
     */
    readonly location: S3Location;
    /**
     * The AWS Glue cataloging options
     */
    readonly catalog?: S3Catalog;
    /**
     * The formatting options for the output files
     */
    readonly formatting?: S3OutputFormatting;
}
export declare class S3Destination implements IDestination {
    private readonly props;
    readonly connectorType: ConnectorType;
    private readonly compatibleFlows;
    constructor(props: S3DestinationProps);
    private buildAggregationConfig;
    private buildPrefixConfig;
    bind(flow: IFlow): CfnFlow.DestinationFlowConfigProperty;
    /**
     * see: https://docs.aws.amazon.com/appflow/latest/userguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-access-gdc
     * see: https://docs.aws.amazon.com/appflow/latest/userguide/security_iam_service-role-policies.html#access-gdc
     * @param flow
     * @param database - the AWS Glue database
     * @param tablePrefix - the prefix for the tables that will be created in the AWS Glue database
     * @returns a tuple consisting of a role for cataloguing with a specified policy
     */
    private buildRoleAndPolicyForCatalog;
    private buildDestinationConnectorProperties;
    private tryAddNodeDependency;
}
