"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mapping = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const operation_1 = require("./operation");
const tasks_1 = require("./tasks");
const TT_MAPALL = "Map_all";
const TT_MAP = "Map";
const TT_MERGE = "Merge";
const TT_ARITHMETIC = "Arithmetic";
const OP_NOOP = "NO_OP";
const OP_ADDITION = "ADDITION";
const OP_SUBTRACTION = "SUBTRACTION";
const OP_MULTIPLICATION = "MULTIPLICATION";
const OP_DIVISION = "DIVISION";
const TP_SOURCE_DATA_TYPE = "SOURCE_DATA_TYPE";
const TP_DESTINATION_DATA_TYPE = "DESTINATION_DATA_TYPE";
/**
 * A representation of an instance of a mapping operation, that is an operation translating source to destination fields
 */
class Mapping extends operation_1.OperationBase {
    static mapAll(config) {
        return new Mapping([
            new tasks_1.Task(TT_MAPALL, [], { operation: OP_NOOP }, [
                {
                    key: "EXCLUDE_SOURCE_FIELDS_LIST",
                    value: config ? `["${aws_cdk_lib_1.Fn.join('","', config.exclude)}"]` : "[]",
                },
            ]),
        ]);
    }
    static map(from, to) {
        const props = [];
        if (from.dataType) {
            props.push({ key: TP_SOURCE_DATA_TYPE, value: from.dataType });
        }
        if (to.dataType) {
            props.push({ key: TP_DESTINATION_DATA_TYPE, value: to.dataType });
        }
        return new Mapping([
            new tasks_1.Task(TT_MAP, [from.name], { operation: OP_NOOP }, props, to.name),
        ]);
    }
    /**
     * A mapping definition building concatenation of source fields into a destination field
     * @param from an array of source fields
     * @param to a desintation field
     * @param format a format
     * @returns a mapping instance with concatenation definition
     */
    static concat(from, to, format) {
        if (!to.dataType) {
            throw new Error("dataType for 'to' required");
        }
        const tmpField = from.map((f) => f.name).join(",");
        return new Mapping([
            new tasks_1.Task(TT_MERGE, from.map((f) => f.name), { operation: OP_NOOP }, [{ key: "CONCAT_FORMAT", value: format }], tmpField),
            new tasks_1.Task(TT_MAP, [tmpField], { operation: OP_NOOP }, [
                { key: "DESTINATION_DATA_TYPE", value: to.dataType },
                { key: "SOURCE_DATA_TYPE", value: "string" },
            ]),
        ]);
    }
    /**
     * Specifies an addition mapping of two numeric values from asource to a destination
     * @param sourceField1 a numeric value
     * @param sourceField2 a numeric value
     * @param to a numeric value
     * @returns an IMapping instance
     */
    static add(sourceField1, sourceField2, to) {
        return Mapping.arithmetic(sourceField1, sourceField2, to, OP_ADDITION);
    }
    /**
     * Specifies a multiplication mapping of two numeric values from a source to a destination
     * @param sourceField1 a numeric value
     * @param sourceField2 a numeric value
     * @param to a numeric value
     * @returns an IMapping instance
     */
    static multiply(sourceField1, sourceField2, to) {
        return Mapping.arithmetic(sourceField1, sourceField2, to, OP_MULTIPLICATION);
    }
    /**
     * Specifies a subtraction mapping of two numeric values from a source to a destination
     * @param sourceField1 a numeric value
     * @param sourceField2 a numeric value
     * @param to a numeric value
     * @returns an IMapping instance
     */
    static subtract(sourceField1, sourceField2, to) {
        return Mapping.arithmetic(sourceField1, sourceField2, to, OP_SUBTRACTION);
    }
    /**
     * Specifies a division mapping of two numeric values from a source to a destination
     * @param sourceField1 a numeric value
     * @param sourceField2 a numeric value
     * @param to a numeric value
     * @returns an IMapping instance
     */
    static divide(sourceField1, sourceField2, to) {
        return Mapping.arithmetic(sourceField1, sourceField2, to, OP_DIVISION);
    }
    /**
     * Specifies an arithmetic mapping of two numeric values from a source to a destination
     * @param sourceField1 a numeric value
     * @param sourceField2 a numeric value
     * @param to a numeric value
     * @returns an IMapping instance
     */
    static arithmetic(sourceField1, sourceField2, to, operation) {
        if (!to.dataType) {
            throw new Error("dataType for 'to' required");
        }
        const tmpField = `${sourceField1.name},${sourceField2.name}`;
        return new Mapping([
            new tasks_1.Task(TT_ARITHMETIC, [sourceField1.name, sourceField2.name], { operation: operation }, [
                {
                    key: "MATH_OPERATION_FIELDS_ORDER",
                    value: tmpField,
                },
            ], tmpField),
            new tasks_1.Task(TT_MAP, [tmpField], { operation: OP_NOOP }, [
                { key: "DESTINATION_DATA_TYPE", value: to.dataType },
                { key: "SOURCE_DATA_TYPE", value: "string" },
            ], to.name),
        ]);
    }
}
exports.Mapping = Mapping;
_a = JSII_RTTI_SYMBOL_1;
Mapping[_a] = { fqn: "@cdklabs/cdk-appflow.Mapping", version: "0.2.1" };
//# sourceMappingURL=data:application/json;base64,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