"use strict";
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = handler;
const API = require("./api");
/* eslint-disable no-console */
async function handler(event) {
    switch (event.RequestType) {
        case "Create":
        case "Update":
            return updateFlowTime(event);
        case "Delete":
            return;
        default:
            throw new Error("Unknown signal");
    }
}
async function updateFlowTime(event) {
    let expressionString = event.ResourceProperties[API.PROP_SCHEDULE];
    const propStartTime = event.ResourceProperties[API.PROP_STARTTIME];
    let startTime = propStartTime
        ? new Date(Date.parse(propStartTime))
        : undefined;
    const propEndTime = event.ResourceProperties[API.PROP_ENDTIME];
    let endTime = propEndTime ? new Date(Date.parse(propEndTime)) : undefined;
    if (/^rate/.test(expressionString)) {
        ({ expressionString, startTime } = buildForRateBasedSchedule(expressionString, startTime));
    }
    else if (/^cron/.test(expressionString)) {
        // do nothing, as we don't change the expressionString
        // TODO: do we have to set the startTime? I think we do
    }
    else {
        throw new Error(`Unrecognized expression ${expressionString}`);
    }
    return {
        Data: {
            [API.ATTR_SCHEDULE]: expressionString,
            [API.ATTR_STARTTIME]: startTime && startTime.getTime() / 1000,
            [API.ATTR_ENDTIME]: endTime && endTime.getTime() / 1000,
        },
    };
}
function buildForRateBasedSchedule(expressionString, startTime) {
    // TODO: the below statement is from a customer. I need to check it more thoroughly
    // Rebuilding expression string as Flows require always plural, so: hour -> hours, etc.
    const matches = /\((\d*)(.*)\)$/.exec(expressionString);
    if (!matches) {
        throw new Error(`Expression ${expressionString} is not a rate-based expression`);
    }
    const rate = parseFloat(matches[1]);
    let rateUnit = matches[2].trim();
    if (!rateUnit.endsWith("s")) {
        rateUnit = `${rateUnit}s`;
    }
    expressionString = `rate(${rate}${rateUnit})`;
    if (startTime) {
        const nowMillis = Date.now();
        const startMillis = startTime.getTime();
        if (nowMillis > startMillis) {
            let denominator = 1;
            switch (rateUnit) {
                case "minutes":
                    denominator = 60 * 1000; // minutes in millis
                    break;
                case "hours":
                    denominator = 60 * 60 * 1000; // hours in millis
                    break;
                case "days":
                    denominator = 24 * 60 * 60 * 1000; // days in millis
                    break;
                default:
                    throw new Error(`Unable to use schedule "${expressionString}"`);
            }
            const cyclesBetween = Math.ceil((nowMillis - startMillis) / denominator / rate);
            startTime = new Date((startMillis / denominator + cyclesBetween * rate) * denominator);
        }
    }
    return { expressionString, startTime };
}
//# sourceMappingURL=data:application/json;base64,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