"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppFlowPermissionsManager = void 0;
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const glob = global;
/**
 * see: https://docs.aws.amazon.com/appflow/latest/userguide/salesforce.html#salesforce-setup
 * @internal
 */
class AppFlowPermissionsManager {
    /**
     * Singleton instance of the AppFlowPermissionsManager
     */
    static instance() {
        if (!glob.__cdkAppFlowPermissionsManager) {
            glob.__cdkAppFlowPermissionsManager = new AppFlowPermissionsManager();
        }
        return glob.__cdkAppFlowPermissionsManager;
    }
    constructor() {
        this.principal = new aws_iam_1.ServicePrincipal("appflow.amazonaws.com");
        this.tokenPermissions = new Map();
        this.writeS3BucketActions = [
            "s3:PutObject",
            "s3:AbortMultipartUpload",
            "s3:ListMultipartUploadParts",
            "s3:ListBucketMultipartUploads",
            "s3:GetBucketAcl",
            "s3:PutObjectAcl",
        ];
        this.readS3BucketActions = ["s3:GetObject", "s3:ListBucket"];
        this.encryptKMSKeyActions = [
            "kms:Encrypt",
            "kms:GenerateDataKey",
        ];
        this.decryptKMSKeyActions = ["kms:Decrypt"];
        this.readSMSecretActions = [
            "secretsmanager:GetSecretValue",
            "secretsmanager:DescribeSecret",
        ];
    }
    getActionsFor(arn) {
        let actions;
        let isNew;
        if (this.tokenPermissions.has(arn)) {
            actions = this.tokenPermissions.get(arn);
            isNew = false;
        }
        else {
            actions = new Set();
            this.tokenPermissions.set(arn, actions);
            isNew = true;
        }
        return { actions, isNew: isNew };
    }
    grantBucketWrite(bucket) {
        if (bucket) {
            const { actions, isNew } = this.getActionsFor(bucket.bucketArn);
            this.writeS3BucketActions.forEach((a) => actions.add(a));
            if (isNew) {
                const statement = new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    principals: [this.principal],
                    actions: aws_cdk_lib_1.Lazy.list({ produce: () => Array.from(actions) }),
                    resources: [bucket.bucketArn, bucket.arnForObjects("*")],
                    conditions: {
                        StringEquals: {
                            "aws:SourceAccount": aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        },
                    },
                });
                const addResult = bucket.addToResourcePolicy(statement);
                if (!addResult.statementAdded) {
                    aws_cdk_lib_1.Annotations.of(bucket).addWarning("Cannot change bucket policy of imported S3 bucket. Ensure that your bucket policy contains appropriate permissions");
                }
            }
            this.grantKeyEncrypt(bucket.encryptionKey);
        }
    }
    grantBucketRead(bucket) {
        if (bucket) {
            const { actions, isNew } = this.getActionsFor(bucket.bucketArn);
            this.readS3BucketActions.forEach((a) => actions.add(a));
            if (isNew) {
                const statement = new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    principals: [this.principal],
                    actions: aws_cdk_lib_1.Lazy.list({ produce: () => Array.from(actions) }),
                    resources: [bucket.bucketArn, bucket.arnForObjects("*")],
                    conditions: {
                        StringEquals: {
                            "aws:SourceAccount": aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        },
                    },
                });
                const addResult = bucket.addToResourcePolicy(statement);
                if (!addResult.statementAdded) {
                    aws_cdk_lib_1.Annotations.of(bucket).addWarning("Cannot change bucket policy of imported S3 bucket. Ensure that your bucket policy contains appropriate permissions");
                }
            }
            this.grantKeyDecrypt(bucket.encryptionKey);
        }
    }
    grantBucketReadWrite(bucket) {
        if (bucket) {
            this.grantBucketRead(bucket);
            this.grantBucketWrite(bucket);
        }
    }
    grantKeyEncryptDecrypt(key) {
        this.grantKeyEncrypt(key);
        this.grantKeyDecrypt(key);
    }
    grantKeyEncrypt(key) {
        if (key) {
            const { actions, isNew } = this.getActionsFor(key.keyArn);
            this.encryptKMSKeyActions.forEach((a) => actions.add(a));
            if (isNew) {
                const statement = new aws_iam_1.PolicyStatement({
                    principals: [this.principal],
                    actions: aws_cdk_lib_1.Lazy.list({ produce: () => Array.from(actions) }),
                    resources: ["*"],
                    conditions: {
                        StringEquals: {
                            "aws:SourceAccount": aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        },
                    },
                });
                const addResult = key.addToResourcePolicy(statement, 
                /* allowNoOp */ true);
                if (!addResult.statementAdded) {
                    aws_cdk_lib_1.Annotations.of(key).addWarning("Cannot change key policy of imported kms key. Ensure that your key policy contains appropriate permissions");
                }
            }
        }
    }
    grantKeyDecrypt(key) {
        if (key) {
            const { actions, isNew } = this.getActionsFor(key.keyArn);
            this.decryptKMSKeyActions.forEach((a) => actions.add(a));
            if (isNew) {
                const statement = new aws_iam_1.PolicyStatement({
                    principals: [this.principal],
                    actions: aws_cdk_lib_1.Lazy.list({ produce: () => Array.from(actions) }),
                    resources: ["*"],
                    conditions: {
                        StringEquals: {
                            "aws:SourceAccount": aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        },
                    },
                });
                const addResult = key.addToResourcePolicy(statement, 
                /* allowNoOp */ true);
                if (!addResult.statementAdded) {
                    aws_cdk_lib_1.Annotations.of(key).addWarning("Cannot change key policy of imported kms key. Ensure that your key policy contains appropriate permissions");
                }
            }
        }
    }
    grantSecretRead(secret) {
        if (secret) {
            const { actions, isNew } = this.getActionsFor(secret.secretArn);
            this.readSMSecretActions.forEach((a) => actions.add(a));
            if (isNew) {
                const statement = new aws_iam_1.PolicyStatement({
                    principals: [this.principal],
                    actions: aws_cdk_lib_1.Lazy.list({ produce: () => Array.from(actions) }),
                    resources: [
                        secret.secretFullArn
                            ? secret.secretFullArn
                            : `${secret.secretArn}-??????`,
                    ],
                    conditions: {
                        StringEquals: {
                            "aws:SourceAccount": aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        },
                    },
                });
                const addResult = secret.addToResourcePolicy(statement);
                if (!addResult.statementAdded) {
                    aws_cdk_lib_1.Annotations.of(secret).addWarning("Cannot change secret policy of imported sm secret. Ensure that your secret policy contains appropriate permissions");
                }
                this.grantKeyEncryptDecrypt(secret.encryptionKey);
            }
        }
    }
}
exports.AppFlowPermissionsManager = AppFlowPermissionsManager;
//# sourceMappingURL=data:application/json;base64,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