"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SnsPublish = exports.MessageAttributeDataType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * The data type set for the SNS message attributes
 *
 * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
 */
var MessageAttributeDataType;
(function (MessageAttributeDataType) {
    /**
     * Strings are Unicode with UTF-8 binary encoding
     */
    MessageAttributeDataType["STRING"] = "String";
    /**
     * An array, formatted as a string
     *
     * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
     */
    MessageAttributeDataType["STRING_ARRAY"] = "String.Array";
    /**
     * Numbers are positive or negative integers or floating-point numbers
     *
     * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
     */
    MessageAttributeDataType["NUMBER"] = "Number";
    /**
     * Binary type attributes can store any binary data
     *
     * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
     */
    MessageAttributeDataType["BINARY"] = "Binary";
})(MessageAttributeDataType = exports.MessageAttributeDataType || (exports.MessageAttributeDataType = {}));
/**
 * A Step Functions Task to publish messages to SNS topic.
 *
 */
class SnsPublish extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _b;
        super(scope, id, props);
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_SnsPublishProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.integrationPattern = (_b = props.integrationPattern) !== null && _b !== void 0 ? _b : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SnsPublish.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            if (!sfn.FieldUtils.containsTaskToken(props.message)) {
                throw new Error('Task Token is required in `message` Use JsonPath.taskToken to set the token.');
            }
        }
        this.taskPolicies = [
            new iam.PolicyStatement({
                actions: ['sns:Publish'],
                resources: [this.props.topic.topicArn],
            }),
        ];
    }
    /**
     * Provides the SNS Publish service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sns', 'publish', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                TopicArn: this.props.topic.topicArn,
                Message: this.props.message.value,
                MessageStructure: this.props.messagePerSubscriptionType ? 'json' : undefined,
                MessageAttributes: renderMessageAttributes(this.props.messageAttributes),
                Subject: this.props.subject,
            }),
        };
    }
}
exports.SnsPublish = SnsPublish;
_a = JSII_RTTI_SYMBOL_1;
SnsPublish[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.SnsPublish", version: "1.156.1" };
SnsPublish.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
function renderMessageAttributes(attributes) {
    if (attributes === undefined) {
        return undefined;
    }
    const renderedAttributes = {};
    Object.entries(attributes).map(([key, val]) => {
        renderedAttributes[key] = renderMessageAttributeValue(val);
    });
    return sfn.TaskInput.fromObject(renderedAttributes).value;
}
function renderMessageAttributeValue(attribute) {
    const dataType = attribute.dataType;
    if (attribute.value instanceof sfn.TaskInput) {
        return {
            DataType: dataType !== null && dataType !== void 0 ? dataType : MessageAttributeDataType.STRING,
            StringValue: dataType !== MessageAttributeDataType.BINARY ? attribute.value.value : undefined,
            BinaryValue: dataType === MessageAttributeDataType.BINARY ? attribute.value.value : undefined,
        };
    }
    if (dataType === MessageAttributeDataType.BINARY) {
        return { DataType: dataType, BinaryValue: `${attribute.value}` };
    }
    if (core_1.Token.isUnresolved(attribute.value)) {
        return { DataType: dataType !== null && dataType !== void 0 ? dataType : MessageAttributeDataType.STRING, StringValue: attribute.value };
    }
    validateMessageAttribute(attribute);
    if (Array.isArray(attribute.value)) {
        return { DataType: MessageAttributeDataType.STRING_ARRAY, StringValue: JSON.stringify(attribute.value) };
    }
    const value = attribute.value;
    if (typeof value === 'number') {
        return { DataType: MessageAttributeDataType.NUMBER, StringValue: `${value}` };
    }
    else {
        return { DataType: MessageAttributeDataType.STRING, StringValue: `${value}` };
    }
}
function validateMessageAttribute(attribute) {
    const dataType = attribute.dataType;
    const value = attribute.value;
    if (dataType === undefined) {
        return;
    }
    if (Array.isArray(value)) {
        if (dataType !== MessageAttributeDataType.STRING_ARRAY) {
            throw new Error(`Requested SNS message attribute type was ${dataType} but ${value} was of type Array`);
        }
        const validArrayTypes = ['string', 'boolean', 'number'];
        value.forEach((v) => {
            if (v !== null || !validArrayTypes.includes(typeof v)) {
                throw new Error(`Requested SNS message attribute type was ${typeof value} but Array values must be one of ${validArrayTypes}`);
            }
        });
        return;
    }
    const error = new Error(`Requested SNS message attribute type was ${dataType} but ${value} was of type ${typeof value}`);
    switch (typeof value) {
        case 'string':
            // trust the user or will default to string
            if (sfn.JsonPath.isEncodedJsonPath(attribute.value)) {
                return;
            }
            if (dataType === MessageAttributeDataType.STRING ||
                dataType === MessageAttributeDataType.BINARY) {
                return;
            }
            throw error;
        case 'number':
            if (dataType === MessageAttributeDataType.NUMBER) {
                return;
            }
            throw error;
        case 'boolean':
            if (dataType === MessageAttributeDataType.STRING) {
                return;
            }
            throw error;
        default:
            throw error;
    }
}
//# sourceMappingURL=data:application/json;base64,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