"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __exportStar = (this && this.__exportStar) || function(m, exports) {
    for (var p in m) if (p !== "default" && !exports.hasOwnProperty(p)) __createBinding(exports, m, p);
};
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKPipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const pdk_nag_1 = require("../../pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codecommit_1 = require("aws-cdk-lib/aws-codecommit");
const aws_codepipeline_1 = require("aws-cdk-lib/aws-codepipeline");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const cdk_nag_1 = require("cdk-nag");
const sonar_code_scanner_1 = require("./code_scanner/sonar-code-scanner");
__exportStar(require("./code_scanner/sonar-code-scanner"), exports);
const DEFAULT_BRANCH_NAME = "mainline";
/**
 * An extension to CodePipeline which configures sane defaults for a NX Monorepo
 * codebase. In addition to this, it also creates a CodeCommit repository with
 * automated PR builds and approvals.
 */
class PDKPipeline extends pipelines_1.CodePipeline {
    constructor(scope, id, props) {
        const codeRepository = new aws_codecommit_1.Repository(scope, "CodeRepository", {
            repositoryName: props.repositoryName,
        });
        codeRepository.applyRemovalPolicy(props.codeCommitRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.RETAIN);
        const accessLogsBucket = new aws_s3_1.Bucket(scope, "AccessLogsBucket", {
            versioned: false,
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
        });
        const artifactBucket = new aws_s3_1.Bucket(scope, "ArtifactsBucket", {
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            encryption: props.crossAccountKeys
                ? aws_s3_1.BucketEncryption.KMS
                : aws_s3_1.BucketEncryption.S3_MANAGED,
            encryptionKey: props.crossAccountKeys
                ? new aws_kms_1.Key(scope, "ArtifactKey", {
                    enableKeyRotation: true,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                })
                : undefined,
            publicReadAccess: false,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            serverAccessLogsPrefix: "access-logs",
            serverAccessLogsBucket: accessLogsBucket,
        });
        const codePipeline = new aws_codepipeline_1.Pipeline(scope, "CodePipeline", {
            enableKeyRotation: props.crossAccountKeys,
            restartExecutionOnUpdate: true,
            crossAccountKeys: props.crossAccountKeys,
            artifactBucket,
        });
        // ignore input and primaryOutputDirectory
        const { input, primaryOutputDirectory, commands, ...synthShellStepPartialProps } = props.synthShellStepPartialProps || {};
        const synthShellStep = new pipelines_1.ShellStep("Synth", {
            input: pipelines_1.CodePipelineSource.codeCommit(codeRepository, props.defaultBranchName || DEFAULT_BRANCH_NAME),
            installCommands: [
                "npm install -g aws-cdk",
                "yarn install --frozen-lockfile || npx projen && yarn install --frozen-lockfile",
            ],
            commands: commands && commands.length > 0
                ? commands
                : ["npx nx run-many --target=build --all"],
            primaryOutputDirectory: props.primarySynthDirectory,
            ...(synthShellStepPartialProps || {}),
        });
        synthShellStep.addOutputDirectory(".");
        const codePipelineProps = {
            codePipeline,
            ...props,
            crossAccountKeys: undefined,
            synth: synthShellStep,
        };
        super(scope, id, codePipelineProps);
        this.id = id;
        this.codeRepository = codeRepository;
        this.sonarCodeScannerConfig = props.sonarCodeScannerConfig
            ? {
                cdkOutDir: props.primarySynthDirectory,
                ...props.sonarCodeScannerConfig,
            }
            : undefined;
        new aws_cdk_lib_1.CfnOutput(scope, "CodeRepositoryGRCUrl", {
            exportName: "CodeRepositoryGRCUrl",
            value: this.codeRepository.repositoryCloneUrlGrc,
        });
    }
    /**
     * @inheritDoc
     */
    addStage(stage, options) {
        // Add any root Aspects to the stage level as currently this doesn't happen automatically
        aws_cdk_lib_1.Aspects.of(stage.node.root).all.forEach((aspect) => aws_cdk_lib_1.Aspects.of(stage).add(aspect));
        return super.addStage(stage, options);
    }
    buildPipeline() {
        super.buildPipeline();
        this.sonarCodeScannerConfig &&
            new sonar_code_scanner_1.SonarCodeScanner(this, "SonarCodeScanner", {
                artifactBucketArn: this.pipeline.artifactBucket.bucketArn,
                artifactBucketKeyArn: this.pipeline.artifactBucket.encryptionKey?.keyArn,
                synthBuildArn: this.synthProject.projectArn,
                ...this.sonarCodeScannerConfig,
            });
        this.suppressCDKViolations();
    }
    suppressCDKViolations() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}CodePipeline/Role/DefaultPolicy/Resource`, [
                {
                    id: RuleId,
                    reason: "Actions contain wildcards which are valid for CodePipeline as all of these operations are required.",
                    appliesTo: [
                        {
                            regex: "/^Action::s3:.*$/g",
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to any and all artifacts in the ArtifactsBucket.",
                    appliesTo: [
                        {
                            regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*$/g",
                        },
                        {
                            regex: "/^Action::kms:ReEncrypt\\*$/g",
                        },
                        {
                            regex: "/^Action::kms:GenerateDataKey\\*$/g",
                        },
                    ],
                },
            ]);
            pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}CodePipeline/Role/DefaultPolicy/Resource`, [
                {
                    id: RuleId,
                    reason: "Actions contain wildcards which are valid for CodePipeline as all of these operations are required.",
                    appliesTo: [
                        {
                            regex: "/^Action::s3:.*$/g",
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to any and all artifacts in the ArtifactsBucket.",
                    appliesTo: [
                        {
                            regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*$/g",
                        },
                    ],
                },
            ]);
            pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}CodePipeline/Source/CodeCommit/CodePipelineActionRole/DefaultPolicy/Resource`, [
                {
                    id: RuleId,
                    reason: "Actions contain wildcards which are valid for CodePipeline as all of these operations are required.",
                    appliesTo: [
                        {
                            regex: "/^Action::s3:.*$/g",
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to any and all artifacts in the ArtifactsBucket.",
                    appliesTo: [
                        {
                            regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*$/g",
                        },
                        {
                            regex: "/^Action::kms:ReEncrypt\\*$/g",
                        },
                        {
                            regex: "/^Action::kms:GenerateDataKey\\*$/g",
                        },
                    ],
                },
            ]);
            pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}CodePipeline/Build/Synth/CdkBuildProject/Role/DefaultPolicy/Resource`, [
                {
                    id: RuleId,
                    reason: "Actions contain wildcards which are valid for CodePipeline as all of these operations are required.",
                    appliesTo: [
                        {
                            regex: "/^Action::s3:.*$/g",
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to any and all artifacts in the ArtifactsBucket.",
                    appliesTo: [
                        {
                            regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*$/g",
                        },
                        {
                            regex: "/^Action::kms:ReEncrypt\\*$/g",
                        },
                        {
                            regex: "/^Action::kms:GenerateDataKey\\*$/g",
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to manage logs and streams whose names are dynamically determined.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/codebuild/<CodePipelineBuildSynthCdkBuildProject.*>:\\*$/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to create report groups that are dynamically determined.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:codebuild:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:report-group/<CodePipelineBuildSynthCdkBuildProject.*>-\\*$/g`,
                        },
                    ],
                },
            ]);
            pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}${this.id}/UpdatePipeline/SelfMutation/Role/DefaultPolicy/Resource`, [
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to manage logs and streams whose names are dynamically determined.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/codebuild/<${this.id}UpdatePipelineSelfMutation.*>:\\*$/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to create report groups that are dynamically determined.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:codebuild:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:report-group/<${this.id}UpdatePipelineSelfMutation.*>-\\*$/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to assume a role from within the current account in order to deploy.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:\\*:iam::${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:role/\\*$/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to any and all artifacts in the ArtifactsBucket.",
                    appliesTo: [
                        {
                            regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*$/g",
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to list all buckets and stacks.",
                    appliesTo: [
                        {
                            regex: "/^Resource::\\*$/g",
                        },
                    ],
                },
            ]);
            pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}${this.id}/UpdatePipeline/SelfMutation/Role/DefaultPolicy/Resource`, [
                {
                    id: RuleId,
                    reason: "Actions contain wildcards which are valid for CodePipeline as all of these operations are required.",
                    appliesTo: [
                        {
                            regex: "/^Action::s3:.*$/g",
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to manage logs and streams whose names are dynamically determined.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:(<AWS::AccountId>|${stack.account}):log-group:/aws/codebuild/<${this.id}UpdatePipelineSelfMutation.*>:\\*$/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to create report groups that are dynamically determined.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:codebuild:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:report-group/<${this.id}UpdatePipelineSelfMutation.*>-\\*$/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to assume a role from within the current account in order to deploy.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:\\*:iam::${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:role/\\*$/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to any and all artifacts in the ArtifactsBucket.",
                    appliesTo: [
                        {
                            regex: "/^Resource::<ArtifactsBucket.*.Arn>/\\*$/g",
                        },
                        {
                            regex: "/^Action::kms:ReEncrypt\\*$/g",
                        },
                        {
                            regex: "/^Action::kms:GenerateDataKey\\*$/g",
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "CodePipeline requires access to list all buckets and stacks.",
                    appliesTo: [
                        {
                            regex: "/^Resource::\\*$/g",
                        },
                    ],
                },
            ]);
            pdk_nag_1.PDKNag.addResourceSuppressionsByPathNoThrow(stack, `${pdk_nag_1.PDKNag.getStackPrefix(stack)}${this.id}/Assets/FileRole/DefaultPolicy/Resource`, [
                {
                    id: RuleId,
                    reason: "Asset role requires access to the Artifacts Bucket",
                },
            ]);
        });
        [
            "AwsSolutions-CB4",
            "AwsPrototyping-CodeBuildProjectKMSEncryptedArtifacts",
        ].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                {
                    id: RuleId,
                    reason: "Encryption of Codebuild is not required.",
                },
            ]);
        });
        ["AwsSolutions-S1", "AwsPrototyping-S3BucketLoggingEnabled"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addStackSuppressions(stack, [
                {
                    id: RuleId,
                    reason: "Access Log buckets should not have s3 bucket logging",
                },
            ]);
        });
    }
}
exports.PDKPipeline = PDKPipeline;
_a = JSII_RTTI_SYMBOL_1;
PDKPipeline[_a] = { fqn: "aws-prototyping-sdk.pipeline.PDKPipeline", version: "0.14.19" };
//# sourceMappingURL=data:application/json;base64,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