"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RuleMetadata = exports.PackName = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const cdk_nag_1 = require("cdk-nag");
/**
 * Pack name
 *
 */
exports.PackName = "AwsPrototyping";
/**
 * Rule metadata
 *
 */
exports.RuleMetadata = [
    {
        info: "The REST API stage is not associated with AWS WAFv2 web ACL.",
        explanation: "AWS WAFv2 is a web application firewall that helps protect web applications and APIs from attacks by allowing configured rules to allow, block, or monitor (count) web requests based on customizable rules and conditions that are defined.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWAssociatedWithWAF,
    },
    {
        info: "The API does not implement authorization.",
        explanation: "In most cases an API needs to have an authentication and authorization implementation strategy. This includes using such approaches as IAM, Cognito User Pools, Custom authorizer, etc.\n\nExample threat: An actor with a network path to an API gateway stage end-point can interact with the API method in question without authorization, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWAuthorization,
    },
    {
        info: "The REST API does not have request validation enabled.",
        explanation: "The API should have basic request validation enabled. If the API is integrated with a custom source (Lambda, ECS, etc..) in the backend, deeper input validation should be considered for implementation.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.apigw.APIGWRequestValidation,
    },
    {
        info: "The Athena workgroup does not encrypt query results.",
        explanation: "Encrypting query results stored in S3 helps secure data to meet compliance requirements for data-at-rest encryption.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.athena.AthenaWorkgroupEncryptedQueryResults,
    },
    {
        info: "The Auto Scaling launch configuration does not have public IP addresses disabled.",
        explanation: "If you configure your Network Interfaces with a public IP address, then the associated resources to those Network Interfaces are reachable from the internet. EC2 resources should not be publicly accessible, as this may allow unintended access to your applications or servers.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.autoscaling.AutoScalingLaunchConfigPublicIpDisabled,
    },
    {
        info: "The Cloud9 instance does not use a no-ingress EC2 instance with AWS Systems Manager.",
        explanation: "SSM adds an additional layer of protection as it allows operators to control access through IAM permissions and does not require opening inbound ports.\n\nExample threat: A global internet-based actor can discover Cloud9 EC2 instances that have public IP addresses and that are exposing SSH, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloud9.Cloud9InstanceNoIngressSystemsManager,
    },
    {
        info: "The prototypes CloudFront distribution has not been configured with geographic restrictions (GeoRestriction)",
        explanation: "Geo restriction should be enabled for the distribution in order to limit the surface area exposed to expected geographies\n\nExample threat: A global internet-based actor can discover prototype web assets that are exposed via CloudFront distributions, which may lead to recon and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionGeoRestrictions,
    },
    {
        info: "The CloudFront distribution allows for SSLv3 or TLSv1 for HTTPS viewer connections.",
        explanation: "Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Help protect viewer connections by specifying a viewer certificate that enforces a minimum of TLSv1.1 or TLSv1.2 in the security policy. Distributions that use that use the default CloudFront viewer certificate or use 'vip' for the SslSupportMethod are non-compliant with this rule, as the minimum security policy is set to TLSv1 regardless of the specified MinimumProtocolVersion",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionHttpsViewerNoOutdatedSSL,
    },
    {
        info: "The CloudFront distributions uses SSLv3 or TLSv1 for communication to the origin.",
        explanation: "Vulnerabilities have been and continue to be discovered in the deprecated SSL and TLS protocols. Using a security policy with minimum TLSv1.1 or TLSv1.2 and appropriate security ciphers for HTTPS helps protect viewer connections.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionNoOutdatedSSL,
    },
    {
        info: "The CloudFront distribution does not use an origin access identity an S3 origin.",
        explanation: "Origin access identities help with security by restricting any direct access to objects through S3 URLs.\n\nExample threat: A global internet-based actor can bypass the CloudFront distribution and associated controls (e.g. geo blocking), which may lead to direct access to static assets hosted on the S3 origin possibly impacting the confidentiality, integrity and availability of the data assets hosted on the S3 origin for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionS3OriginAccessIdentity,
    },
    {
        info: "The CloudFront distribution may require integration with AWS WAF.",
        explanation: "The Web Application Firewall can help protect against application-layer attacks that can compromise the security of the system or place unnecessary load on them.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cloudfront.CloudFrontDistributionWAFIntegration,
    },
    {
        info: "The CodeBuild environment stores sensitive credentials (such as AWS_ACCESS_KEY_ID and/or AWS_SECRET_ACCESS_KEY) as plaintext environment variables.",
        explanation: "Do not store these variables in clear text. Storing these variables in clear text leads to unintended data exposure and unauthorized access.\n\nExample threat: An actor who can view the CodeBuild environment variables can obtain the AWS Access Key and Secret Access Key, which may lead to the actor being able to do anything the AWS keys are authorised to do possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectEnvVarAwsCred,
    },
    {
        info: "The CodeBuild project does not use an AWS KMS key for encryption.",
        explanation: "Using an AWS KMS key helps follow the standard security advice of granting least privilege to objects generated by the project.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectKMSEncryptedArtifacts,
    },
    {
        info: "The CodeBuild project does not use images provided by the CodeBuild service or have a cdk-nag suppression rule explaining the need for a custom image.",
        explanation: "Explaining differences/edits to Docker images helps operators better understand system dependencies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectManagedImages,
    },
    {
        info: "The CodeBuild project has privileged mode enabled.",
        explanation: "Privileged grants elevated rights to the system, which introduces additional risk. Privileged mode should only be set to true only if the build project is used to build Docker images. Otherwise, a build that attempts to interact with the Docker daemon fails.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.codebuild.CodeBuildProjectPrivilegedModeDisabled,
    },
    {
        info: "The Cognito user pool does not have AdvancedSecurityMode set to ENFORCED.",
        explanation: "Advanced security features enable the system to detect and act upon malicious sign-in attempts.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolAdvancedSecurityModeEnforced,
    },
    {
        info: "The API Gateway method does not use a Cognito user pool authorizer.",
        explanation: "API Gateway validates the tokens from a successful user pool authentication, and uses them to grant your users access to resources including Lambda functions, or your own API.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolAPIGWAuthorizer,
    },
    {
        info: "The Cognito user pool is not configured to require MFA.",
        explanation: "Multi-factor authentication (MFA) increases security for the application by adding another authentication method, and not relying solely on user name and password.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolMFA,
    },
    {
        info: "The Cognito identity pool allows for unauthenticated logins and does not have a cdk-nag rule suppression with a reason.",
        explanation: "Applications do not warrant unauthenticated guest access in many cases. Metadata explaining the use case allows for transparency to operators.\n\nExample threat: A global internet-based actor who has discovered a prototype endpoint with Cognito unauthenticated logins can does not need to provide credentials to interact with the endpoint, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolNoUnauthenticatedLogins,
    },
    {
        info: "The Cognito user pool does not have a password policy that minimally specify a password length of at least 8 characters, as well as requiring uppercase, numeric, and special characters.",
        explanation: "Strong password policies increase system security by encouraging users to create reliable and secure passwords.\n\nExample threat: An actor who has discovered a prototype endpoint with Cognito authenticated logins can perform a dictionary or brute force attack to authenticate as an authorized Cognito user, which may lead to the actor being able to do anything the associated Cognito user is authorised to do possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.cognito.CognitoUserPoolStrongPasswordPolicy,
    },
    {
        info: "The DMS replication instance is public.",
        explanation: "DMS replication instances can contain sensitive information and access control is required for such accounts.\n\nExample threat: A global internet-based actor can discover DMS instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.dms.DMSReplicationNotPublic,
    },
    {
        info: "The Document DB cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.documentdb.DocumentDBClusterEncryptionAtRest,
    },
    {
        info: "The Document DB cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MongoDB port 27017).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.documentdb.DocumentDBClusterNonDefaultPort,
    },
    {
        info: "The Document DB cluster does not have the username and password stored in Secrets Manager.",
        explanation: "Secrets Manager enables operators to replace hardcoded credentials in your code, including passwords, with an API call to Secrets Manager to retrieve the secret programmatically. This helps ensure the secret can't be compromised by someone examining system code, because the secret no longer exists in the code. Also, operators can configure Secrets Manager to automatically rotate the secret for you according to a specified schedule. This enables you to replace long-term secrets with short-term ones, significantly reducing the risk of compromise.\n\nExample threat: An actor who can view the DocumentDB configuration can obtain the username and password for the DocumentDB cluster, which may lead to the actor being able to access anything the associated DocumentDB user is authorised to do possibly impacting the confidentiality, integrity and availability of the data assets hosted on the DocumentDB cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.documentdb.DocumentDBCredentialsInSecretsManager,
    },
    {
        info: "The DAX cluster does not have server-side encryption enabled.",
        explanation: "Data in cache, configuration data and log files should be encrypted using Server-Side Encryption in order to protect from unauthorized access to the underlying storage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.dynamodb.DAXEncrypted,
    },
    {
        info: "The EBS volume has encryption disabled.",
        explanation: "With EBS encryption, you aren't required to build, maintain, and secure your own key management infrastructure. EBS encryption uses KMS keys when creating encrypted volumes and snapshots. This helps protect data at rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2EBSVolumeEncrypted,
    },
    {
        info: "The EC2 instance is associated with a public IP address.",
        explanation: "Amazon EC2 instances can contain sensitive information and access control is required for such resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2InstanceNoPublicIp,
    },
    {
        info: "The EC2 instance does not have an instance profile attached.",
        explanation: "EC2 instance profiles pass an IAM role to an EC2 instance. Attaching an instance profile to your instances can assist with least privilege and permissions management.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2InstanceProfileAttached,
    },
    {
        info: "The EC2 instance is not within a VPC.",
        explanation: "Because of their logical isolation, domains that reside within an Amazon VPC have an extra layer of security when compared to domains that use public endpoints.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2InstancesInVPC,
    },
    {
        info: "The EC2 instance allows unrestricted inbound IPv4 TCP traffic on one or more common ports (by default these ports include 20, 21, 3389, 3309, 3306, 4333).",
        explanation: "Not restricting access to ports to trusted sources can lead to attacks against the availability, integrity and confidentiality of systems. By default, common ports which should be restricted include port numbers 20, 21, 3389, 3306, and 4333.\n\nExample threat: A global internet-based actor can discover exposed services (e.g. Telnet, SSH, RDS, MySQL) using their common port numbers, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedCommonPorts,
    },
    {
        info: "The Security Group allows for 0.0.0.0/0 or ::/0 inbound access.",
        explanation: "Large port ranges, when open, expose instances to unwanted attacks. More than that, they make traceability of vulnerabilities very difficult. For instance, your web servers may only require 80 and 443 ports to be open, but not all. One of the most common mistakes observed is when all ports for 0.0.0.0/0 range are open in a rush to access the instance. EC2 instances must expose only to those ports enabled on the corresponding security group level.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses and allow ingress to all internet address or move laterally to non-public EC2 instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedInbound,
    },
    {
        info: "The Security Group allows unrestricted SSH access.",
        explanation: "Not allowing ingress (or remote) traffic from 0.0.0.0/0 or ::/0 to port 22 on your resources helps to restrict remote access.\n\nExample threat: A global internet-based actor can discover EC2 instances that have public IP addresses and allow ingress to all internet address to SSH or move laterally to non-public EC2 instances via SSH, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ec2.EC2RestrictedSSH,
    },
    {
        info: "The Security Group does not have a description.",
        explanation: "Descriptions help simplify operations and remove any opportunities for operator errors.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ec2.EC2SecurityGroupDescription,
    },
    {
        info: "The ECR Repository allows open access.",
        explanation: "Removing * principals in an ECR Repository helps protect against unauthorized access.\n\nExample threat: A global internet-based actor who has discovered a ECR repository can access the container images hosted within the repository, which may lead to information disclosure that aids in the intrusion activities being successful against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.ecr.ECROpenAccess,
    },
    {
        info: "One or more containers in the ECS Task Definition do not have container logging enabled.",
        explanation: "Container logging allows operators to view and aggregate the logs from the container. Containers should use the 'awslogs' driver at a minimum.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSTaskDefinitionContainerLogging,
    },
    {
        info: "The ECS Task Definition includes a container definition that directly specifies environment variables.",
        explanation: "Use secrets to inject environment variables during container startup from AWS Systems Manager Parameter Store or Secrets Manager instead of directly specifying plaintext environment variables. Updates to direct environment variables require operators to change task definitions and perform new deployments.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.ecs.ECSTaskDefinitionNoEnvironmentVariables,
    },
    {
        info: "The EFS does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon Elastic File System (EFS).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.efs.EFSEncrypted,
    },
    {
        info: "The EKS Cluster does not publish 'api', 'audit', 'authenticator, 'controllerManager', and 'scheduler' control plane logs.",
        explanation: "EKS control plane logging provides audit and diagnostic logs directly from the Amazon EKS control plane to CloudWatch Logs in your account. These logs make it easy for you to secure and run your clusters. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format LogExport::<log> for exported logs. Example: appliesTo: ['LogExport::authenticate'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.eks.EKSClusterControlPlaneLogs,
    },
    {
        info: "The EKS cluster's Kubernetes API server endpoint has public access enabled.",
        explanation: "A cluster's Kubernetes API server endpoint should not be publicly accessible from the Internet in order to avoid exposing private data and minimizing security risks. The API server endpoints should only be accessible from within a AWS Virtual Private Cloud (VPC).\n\nExample threat: A global internet-based actor who has discovered a EKS cluster Kubernetes API server endpoint can perform reconnaissance and intrusion activities against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.eks.EKSClusterNoEndpointPublicAccess,
    },
    {
        info: "The ElastiCache cluster is not provisioned in a VPC.",
        explanation: "Provisioning the cluster within a VPC allows for better flexibility and control over the cache clusters security, availability, traffic routing and more.\n\nExample threat: A global internet-based actor can discover the ElastiCache cluster that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheClusterInVPC,
    },
    {
        info: "The ElastiCache cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redis port 6379 and Memcached port 11211).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheClusterNonDefaultPort,
    },
    {
        info: "The ElastiCache Redis cluster does not have both encryption in transit and at rest enabled.",
        explanation: "Encryption in transit helps secure communications to the cluster. Encryption at rest helps protect data at rest from unauthorized access.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheRedisClusterEncryption,
    },
    {
        info: "The ElastiCache Redis cluster does not use Redis AUTH for user authentication.",
        explanation: "Redis authentication tokens enable Redis to require a token (password) before allowing clients to execute commands, thereby improving data security.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticache.ElastiCacheRedisClusterRedisAuth,
    },
    {
        info: "The Elastic Beanstalk environment does not upload EC2 Instance logs to S3.",
        explanation: "Beanstalk environment logs should be retained and uploaded to Amazon S3 in order to keep the logging data for future audits, historical purposes or to track and analyze the EB application environment behavior for a long period of time.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkEC2InstanceLogsToS3,
    },
    {
        info: "The Elastic Beanstalk environment does not have managed updates enabled.",
        explanation: "Enable managed platform updates for beanstalk environments in order to receive bug fixes, software updates and new features. Managed platform updates perform immutable environment updates.\n\nExample threat: An actor with a network path to the Elastic Beanstalk environment can attempt to take advantage of a known vulnerability in a platform component used by Elastic Beanstalk, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkManagedUpdatesEnabled,
    },
    {
        info: "The Elastic Beanstalk environment is not configured to use a specific VPC.",
        explanation: "Use a non-default VPC in order to separate your environment from default resources.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elasticbeanstalk.ElasticBeanstalkVPCSpecified,
    },
    {
        info: "The ALB's HTTP listeners are not configured to redirect to HTTPS.",
        explanation: "To help protect data in transit, ensure that your Application Load Balancer automatically redirects unencrypted HTTP requests to HTTPS. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ALBHttpToHttpsRedirection,
    },
    {
        info: "The CLB does not restrict its listeners to only the SSL and HTTPS protocols.",
        explanation: "Ensure that your Classic Load Balancers (CLBs) are configured with SSL or HTTPS listeners. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.elb.ELBTlsHttpsListenersOnly,
    },
    {
        info: "The EMR cluster does not implement authentication via an EC2 Key Pair or Kerberos.",
        explanation: "SSH clients can use an EC2 key pair to authenticate to cluster instances. Alternatively, with EMR release version 5.10.0 or later, solutions can configure Kerberos to authenticate users and SSH connections to the master node.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMRAuthEC2KeyPairOrKerberos,
    },
    {
        info: "The EMR cluster does not use a security configuration with encryption in transit enabled and configured.",
        explanation: "EMR Clusters should have a method for encrypting data in transit using Transport Layer Security (TLS).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMREncryptionInTransit,
    },
    {
        info: "The EMR cluster does not use a security configuration with local disk encryption enabled.",
        explanation: "Local disk encryption uses a combination of open-source HDFS encryption and LUKS encryption to secure data at rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.emr.EMRLocalDiskEncryption,
    },
    {
        info: "The event bus policy allows for open access.",
        explanation: 'An open policy ("*" principal without a condition) grants anonymous access to an event bus. Use a condition to limit the permission to accounts that fulfill a certain requirement, such as being a member of a certain AWS organization.\n\nExample threat: A global internet-based actor who has discovered the Event Bridge event bus (e.g. Endpoint ID) can put arbitrary events onto the bus, which may lead to which could be processed by the prototype possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype',
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.eventbridge.EventBusOpenAccess,
    },
    {
        info: "The Glue crawler or job does not use a security configuration with CloudWatch Log encryption enabled.",
        explanation: "Enabling encryption at rest helps prevent unauthorized users from getting access to the logging data published to CloudWatch Logs.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.glue.GlueEncryptedCloudWatchLogs,
    },
    {
        info: "The Glue job does not have use a security configuration with job bookmark encryption enabled.",
        explanation: "Job bookmark encryption encrypts bookmark data before it is sent to Amazon S3 for storage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.glue.GlueJobBookmarkEncrypted,
    },
    {
        info: "The IAM user, role, or group uses AWS managed policies.",
        explanation: "An AWS managed policy is a standalone policy that is created and administered by AWS. Currently, many AWS managed policies do not restrict resource scope. Replace AWS managed policies with system specific (customer) managed policies. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format Policy::<policy> for AWS managed policies. Example: appliesTo: ['Policy::arn:<AWS::Partition>:iam::aws:policy/foo'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMNoManagedPolicies,
    },
    {
        info: "The IAM entity contains wildcard permissions and does not have a cdk-nag rule suppression with evidence for those permission.",
        explanation: "Metadata explaining the evidence (e.g. via supporting links) for wildcard permissions allows for transparency to operators. This is a granular rule that returns individual findings that can be suppressed with appliesTo. The findings are in the format Action::<action> for policy actions and Resource::<resource> for resources. Example: appliesTo: ['Action::s3:*'].",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMNoWildcardPermissions,
    },
    {
        info: "The IAM policy grants admin access - meaning the policy allows a principal to perform unlimited actions on any service",
        explanation: "AWS Identity and Access Management (IAM) can help you incorporate the principles of least privilege and separation of duties with access permissions and authorizations, by ensuring that IAM groups have at least one IAM user. Placing IAM users in groups based on their associated permissions or job function is one way to incorporate least privilege.\n\nExample threat: A global internet-based actor who has successfully obtained valid keys or a session associated of the IAM Principal associated with the IAM policy can perform unlimited AWS actions on any AWS service which are exposed via the AWS API/Management Console/CLI, which may lead to broad and unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.iam.IAMPolicyNoStatementsWithAdminAccess,
    },
    {
        info: "The IAM policy grants full access - meaning the policy allows unlimited actions for a given AWS service",
        explanation: "Ensure IAM Actions are restricted to only those actions that are needed. Allowing users to have more privileges than needed to complete a task may violate the principle of least privilege and separation of duties.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.iam.IAMPolicyNoStatementsWithFullAccess,
    },
    {
        info: "The Kinesis Data Firehose delivery stream does not have server-side encryption enabled.",
        explanation: "Enabling encryption allows the system to meet strict regulatory requirements and enhance the security of system data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataFirehoseSSE,
    },
    {
        info: 'The Kinesis Data Stream specifies server-side encryption and does not use the "aws/kinesis" key.',
        explanation: "Customer Managed Keys can incur additional costs that scale with the amount of consumers and producers. Ensure that Customer Managed Keys are required for compliance before using them (https://docs.aws.amazon.com/streams/latest/dev/costs-performance.html).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataStreamDefaultKeyWhenSSE,
    },
    {
        info: "The Kinesis Data Stream does not have server-side encryption enabled.",
        explanation: "Data is encrypted before it's written to the Kinesis stream storage layer, and decrypted after it’s retrieved from storage. This allows the system to meet strict regulatory requirements and enhance the security of system data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kinesis.KinesisDataStreamSSE,
    },
    {
        info: "The KMS Symmetric key does not have automatic key rotation enabled.",
        explanation: "KMS key rotation allow a system to set a rotation schedule for a KMS key so when a AWS KMS key is required to encrypt new data, the KMS service can automatically use the latest version of the HSA backing key to perform the encryption.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.kms.KMSBackingKeyRotationEnabled,
    },
    {
        info: "The Lambda function permission grants public access",
        explanation: "Public access allows anyone on the internet to perform unauthenticated actions on your function and can potentially lead to degraded availability.\n\nExample threat: A global internet-based actor who has discovered the Lambda function name or ARN can invoke/delete/modify the Lambda function without any authentication, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.lambda.LambdaFunctionPublicAccessProhibited,
    },
    {
        info: "The Lambda Function URL allows for public, unauthenticated access.",
        explanation: "AWS Lambda Function URLs allow you to invoke your function via a HTTPS end-point, setting the authentication to NONE allows anyone on the internet to invoke your function.\n\nExample threat: A global internet-based actor who has discovered the Lambda Function URL can invoke the Lambda function without any authentication, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.lambda.LambdaFunctionUrlAuth,
    },
    {
        info: "The non-container Lambda function is not configured to use the latest runtime version.",
        explanation: "Use the latest available runtime for the targeted language to avoid technical debt. Runtimes specific to a language or framework version are deprecated when the version reaches end of life. This rule only applies to non-container Lambda functions.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.lambda.LambdaLatestVersion,
    },
    {
        info: "The MediaStore container does not define a CORS policy.",
        explanation: "Using a CORS policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerCORSPolicy,
    },
    {
        info: "The MediaStore container does not define a container policy.",
        explanation: "Using a container policy helps follow the standard security advice of granting least privilege, or granting only the permissions required to allow needed access to the container.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerHasContainerPolicy,
    },
    {
        info: "The MediaStore container does not require requests to use SSL.",
        explanation: "You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on MediaStore container policies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.mediastore.MediaStoreContainerSSLRequestsOnly,
    },
    {
        info: "The MSK cluster uses plaintext communication between brokers.",
        explanation: "TLS communication secures data-in-transit by encrypting the connection between brokers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.msk.MSKBrokerToBrokerTLS,
    },
    {
        info: "The MSK cluster uses plaintext communication between clients and brokers.",
        explanation: "TLS only communication secures data-in-transit by encrypting the connection between the clients and brokers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.msk.MSKClientToBrokerTLS,
    },
    {
        info: "The Neptune DB instance does have Auto Minor Version Upgrade enabled.",
        explanation: "The Neptune service regularly releases engine updates. Enabling Auto Minor Version Upgrade will allow the service to automatically apply these upgrades to DB Instances.\n\nExample threat: An actor with a network path to the Neptune cluster or instance can attempt to take advantage of a known vulnerability in a component exposed by Neptune, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterAutomaticMinorVersionUpgrade,
    },
    {
        info: "The Neptune DB cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality and prevents unauthorized users from accessing sensitive information.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterEncryptionAtRest,
    },
    {
        info: "The Neptune DB cluster does not have IAM Database Authentication enabled.",
        explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the cluster.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.neptune.NeptuneClusterIAMAuth,
    },
    {
        info: "The OpenSearch Service domain does not only grant access via allowlisted IP addresses.",
        explanation: "Using allowlisted IP addresses helps protect the domain against unauthorized access.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchAllowlistedIPs,
    },
    {
        info: "The OpenSearch Service domain does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist and to help protect data at rest, ensure encryption is enabled for your Amazon OpenSearch Service (OpenSearch Service) domains.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchEncryptedAtRest,
    },
    {
        info: "The OpenSearch Service domain is not provisioned inside a VPC.",
        explanation: "Provisioning the domain within a VPC enables better flexibility and control over the clusters access and security as this feature keeps all traffic between the VPC and OpenSearch domains within the AWS network instead of going over the public Internet.\n\nExample threat: A global internet-based actor can discover the OpenSearch Service domain that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.opensearch.OpenSearchInVPCOnly,
    },
    {
        info: "The OpenSearch Service domain does not have node-to-node encryption enabled.",
        explanation: "Because sensitive data can exist, enable encryption in transit to help protect that data within your Amazon OpenSearch Service (OpenSearch Service) domains.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.opensearch.OpenSearchNodeToNodeEncryption,
    },
    {
        info: "The OpenSearch Service domain does not allow for unsigned requests or anonymous access.",
        explanation: "Restricting public access helps prevent unauthorized access and prevents any unsigned requests to be made to the resources.\n\nExample threat: An actor with a network path to the OpenSearch Service domain can directly access the domain without authentication, which may lead to allowing access to data hosted within the domain possibly impacting the confidentiality, integrity and availability of the data assets hosted on the OpenSearch Service domain for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.opensearch.OpenSearchNoUnsignedOrAnonymousAccess,
    },
    {
        info: "The Quicksight data sources connection is not configured to use SSL.",
        explanation: "SSL secures communications to data sources, especially when using public networks. Using SSL with QuickSight requires the use of certificates signed by a publicly-recognized certificate authority.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.quicksight.QuicksightSSLConnections,
    },
    {
        info: "The RDS Aurora MySQL/PostgresSQL cluster does not have IAM Database Authentication enabled.",
        explanation: "With IAM Database Authentication enabled, the system doesn't have to use a password when connecting to the MySQL/PostgreSQL database instances, instead it uses an authentication token.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.AuroraMySQLPostgresIAMAuth,
    },
    {
        info: "The RDS DB instance does not have automatic minor version upgrades enabled.",
        explanation: "Enable automatic minor version upgrades on your Amazon Relational Database Service (RDS) instances to ensure the latest minor version updates to the Relational Database Management System (RDBMS) are installed, which may include security patches and bug fixes.\n\nExample threat: An actor with a network path to the RDS cluster or instance can attempt to take advantage of a known vulnerability in a component exposed by RDS, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSAutomaticMinorVersionUpgradeEnabled,
    },
    {
        info: "The RDS DB instance allows public access.",
        explanation: "Amazon RDS database instances can contain sensitive information, hence appropriate access control and principles of least privilege should be applied.\n\nExample threat: A global internet-based actor who has discovered the RDS DB instance endpoint can perform reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSInstancePublicAccess,
    },
    {
        info: "The RDS instance or Aurora DB cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. MySQL/Aurora port 3306, SQL Server port 1433, PostgreSQL port 5432, etc).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.RDSNonDefaultPort,
    },
    {
        info: "The RDS DB Security Group allows for 0.0.0.0/0 inbound access.",
        explanation: "RDS DB security groups should not allow access from 0.0.0.0/0 (i.e. anywhere, every machine that has the ability to establish a connection) in order to reduce the risk of unauthorized access.\n\nExample threat: A global internet-based actor can discover RDS DB instances that have public IP addresses and allow ingress to all internet address or move laterally to non-public RDS DB instances, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data assets hosted on the RDS Cluster or instance for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.rds.RDSRestrictedInbound,
    },
    {
        info: "The RDS DB instance or Aurora DB cluster does not have encryption at rest enabled.",
        explanation: "Because sensitive data can exist at rest in Amazon RDS DB instances, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.rds.RDSStorageEncrypted,
    },
    {
        info: "The Redshift cluster does not have encryption at rest enabled.",
        explanation: "Encrypting data-at-rest protects data confidentiality.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterEncryptionAtRest,
    },
    {
        info: "The Redshift cluster is not provisioned in a VPC.",
        explanation: "Provisioning the cluster within a VPC allows for better flexibility and control over the Redshift clusters security, availability, traffic routing and more.\n\nExample threat: A global internet-based actor can discover a RedShift cluster that have public IP addresses, which may lead to reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data with the cluster used within the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterInVPC,
    },
    {
        info: "The Redshift cluster uses the default endpoint port.",
        explanation: "Port obfuscation (using a non default endpoint port) adds an additional layer of defense against non-targeted attacks (i.e. Redshift port 5439).",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterNonDefaultPort,
    },
    {
        info: 'The Redshift cluster uses the default "awsuser" username.',
        explanation: 'Using a custom user name instead of the default master user name (i.e. "awsuser") provides an additional layer of defense against non-targeted attacks.',
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterNonDefaultUsername,
    },
    {
        info: "The Redshift cluster allows public access.",
        explanation: "Amazon Redshift clusters can contain sensitive information, hence appropriate access control and principles of least privilege should be applied.\n\nExample threat: A global internet-based actor who has discovered the Redshift cluster endpoint can perform reconnaissance and intrusion activities (e.g. brute force/dictionary attack to authenticate as a valid user) against the exposed attack surface, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the Redshift cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterPublicAccess,
    },
    {
        info: "The Redshift cluster does not have version upgrade enabled.",
        explanation: "Version Upgrade must enabled on the cluster in order to automatically receive upgrades during the maintenance window.\n\nExample threat: An actor with a network path to the Redshift cluster can attempt to take advantage of a known vulnerability in a component exposed by Redshift, which may lead to unknown impacts possibly impacting the confidentiality, integrity and availability of the data assets hosted on the Redshift cluster for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.redshift.RedshiftClusterVersionUpgrade,
    },
    {
        info: "The Redshift cluster does not require TLS/SSL encryption.",
        explanation: "Ensure that your Amazon Redshift clusters require TLS/SSL encryption to connect to SQL clients. Because sensitive data can exist, enable encryption in transit to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.redshift.RedshiftRequireTlsSSL,
    },
    {
        info: "The S3 bucket does not prohibit public access through bucket level settings.",
        explanation: "Keep sensitive data safe from unauthorized remote users by preventing public access at the bucket level.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public read or write can read or write data to or from the S3 bucket, which may lead to possibly impacting the confidentiality, integrity and availability of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketLevelPublicAccessProhibited,
    },
    {
        info: "The S3 Bucket does not have server access logs enabled.",
        explanation: "Amazon Simple Storage Service (Amazon S3) server access logging provides a method to monitor the network for potential cybersecurity events. The events are monitored by capturing detailed records for the requests that are made to an Amazon S3 bucket. Each access log record provides details about a single access request. The details include the requester, bucket name, request time, request action, response status, and an error code, if relevant.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketLoggingEnabled,
    },
    {
        info: "The S3 Bucket does not prohibit public read access through its Block Public Access configurations and bucket ACLs.",
        explanation: "The management of access should be consistent with the classification of the data.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public read can read data from the S3 bucket, which may lead to possibly impacting the confidentiality of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketPublicReadProhibited,
    },
    {
        info: "The S3 Bucket does not prohibit public write access through its Block Public Access configurations and bucket ACLs.",
        explanation: "The management of access should be consistent with the classification of the data.\n\nExample threat: A global internet-based actor who has discovered a S3 bucket configured for public write can write data to, or overwrite data within the S3 bucket, which may lead to possibly impacting the integrity and availability of the data assets hosted on the S3 bucket for the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3BucketPublicWriteProhibited,
    },
    {
        info: "The S3 Bucket or bucket policy does not require requests to use SSL/TLS.",
        explanation: "You can use HTTPS (TLS) to help prevent potential attackers from eavesdropping on or manipulating network traffic using person-in-the-middle or similar attacks. You should allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition on Amazon S3 bucket policies.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.s3.S3BucketSSLRequestsOnly,
    },
    {
        info: "The S3 static website bucket either has an open world bucket policy or does not use a CloudFront Origin Access Identity (OAI) in the bucket policy for limited getObject and/or putObject permissions.",
        explanation: "An OAI allows you to provide access to content in your S3 static website bucket through CloudFront URLs without enabling public access through an open bucket policy, disabling S3 Block Public Access settings, and/or through object ACLs.\n\nExample threat: A global internet-based actor who has discovered a S3 hosted website can discover prototype web assets that are hosted on the website, which may lead to recon and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.s3.S3WebBucketOAIAccess,
    },
    {
        info: "The SageMaker notebook instance does not have an encrypted storage volume.",
        explanation: "Encrypting storage volumes helps protect SageMaker data-at-rest.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookInstanceKMSKeyConfigured,
    },
    {
        info: "The SageMaker notebook instance is not provisioned inside a VPC.",
        explanation: "Provisioning the notebook instances inside a VPC enables the notebook to access VPC-only resources such as EFS file systems.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookInVPC,
    },
    {
        info: "The SageMaker notebook does not disable direct internet access.",
        explanation: "By preventing direct internet access, you can keep sensitive data from being accessed by unauthorized users.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sagemaker.SageMakerNotebookNoDirectInternetAccess,
    },
    {
        info: "The secret does not have automatic rotation scheduled.",
        explanation: "Rotating secrets on a regular schedule can shorten the period a secret is active, and potentially reduce the business impact if the secret is compromised.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.secretsmanager.SecretsManagerRotationEnabled,
    },
    {
        info: "The SNS topic does not have KMS encryption enabled.",
        explanation: "To help protect data at rest, ensure that your Amazon Simple Notification Service (Amazon SNS) topics require encryption using AWS Key Management Service (AWS KMS). Because sensitive data can exist at rest in published messages, enable encryption at rest to help protect that data.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sns.SNSEncryptedKMS,
    },
    {
        info: "The SNS Topic does not require publishers to use SSL.",
        explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition and the 'sns:Publish' action in the topic policy to force publishers to use SSL. If SSE is already enabled then this control is auto enforced.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sns.SNSTopicSSLPublishOnly,
    },
    {
        info: "The SQS Queue does not have server-side encryption enabled.",
        explanation: "Server side encryption adds additional protection of sensitive data delivered as messages to subscribers.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sqs.SQSQueueSSE,
    },
    {
        info: "The SQS queue does not require requests to use SSL.",
        explanation: "Without HTTPS (TLS), a network-based attacker can eavesdrop on network traffic or manipulate it, using an attack such as man-in-the-middle. Allow only encrypted connections over HTTPS (TLS) using the aws:SecureTransport condition in the queue policy to force requests to use SSL.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.sqs.SQSQueueSSLRequestsOnly,
    },
    {
        info: "The Timestream database does not use a Customer Managed KMS Key for at rest encryption.",
        explanation: "All Timestream tables in a database are encrypted at rest by default using an AWS Managed Key. These keys are rotated every three years. Data at rest must be encrypted using CMKs if you require more control over the permissions and lifecycle of your keys, including the ability to have them automatically rotated on an periodic basis.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.timestream.TimestreamDatabaseCustomerManagedKey,
    },
    {
        info: "A Network ACL or Network ACL entry has been implemented.",
        explanation: "Network ACLs should be used sparingly for the following reasons: they can be complex to manage, they are stateless, every IP address must be explicitly opened in each (inbound/outbound) direction, and they affect a complete subnet. Use security groups when possible as they are stateful and easier to manage.",
        level: cdk_nag_1.NagMessageLevel.WARN,
        rule: cdk_nag_1.rules.vpc.VPCNoNACLs,
    },
    {
        info: "The subnet auto-assigns public IP addresses.",
        explanation: "Manage access to the AWS Cloud by ensuring Amazon Virtual Private Cloud (VPC) subnets are not automatically assigned a public IP address. Amazon Elastic Compute Cloud (EC2) instances that are launched into subnets that have this attribute enabled have a public IP address assigned to their primary network interface.\n\nExample threat: A global internet-based actor can discover VPC-attached resources (e.g. EC2 instances) within the subnet in question that have public IP addresses, which may lead to reconnaissance and intrusion activities being performed against the exposed attack surface possibly impacting the confidentiality, integrity and availability of the data and resource assets associated with the prototype",
        level: cdk_nag_1.NagMessageLevel.ERROR,
        rule: cdk_nag_1.rules.vpc.VPCSubnetAutoAssignPublicIpDisabled,
    },
];
//# sourceMappingURL=data:application/json;base64,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