/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
export declare namespace Nx {
    /**
     * Configuration for nx targetDependencies.
     */
    type TargetDependencies = {
        [target: string]: TargetDependency[];
    };
    /**
     * Configuration for project specific targets.
     */
    type ProjectTargets = {
        [target: string]: ProjectTarget;
    };
    /**
     * Project Target.
     */
    interface ProjectTarget {
        /**
         * List of inputs to hash for cache key, relative to the root of the monorepo.
         *
         * note: must start with leading /
         */
        readonly inputs?: string[];
        /**
         * List of outputs to cache, relative to the root of the monorepo.
         *
         * note: must start with leading /
         */
        readonly outputs?: string[];
        /**
         * List of Target Dependencies.
         */
        readonly dependsOn?: TargetDependency[];
    }
    /**
     * Implicit Dependencies map.
     */
    type ImplicitDependencies = {
        [pkg: string]: string[];
    };
    /**
     * Supported enums for a TargetDependency.
     */
    enum TargetDependencyProject {
        /**
         * Only rely on the package where the target is called.
         *
         * This is usually done for test like targets where you only want to run unit
         * tests on the target packages without testing all dependent packages.
         */
        SELF = "self",
        /**
         * Target relies on executing the target against all dependencies first.
         *
         * This is usually done for build like targets where you want to build all
         * dependant projects first.
         */
        DEPENDENCIES = "dependencies"
    }
    /**
     * Represents an NX Target Dependency.
     */
    interface TargetDependency {
        /**
         * Projen target i.e: build, test, etc
         */
        readonly target: string;
        /**
         * Target dependencies.
         */
        readonly projects: TargetDependencyProject;
    }
    /**
     * Named inputs config
     * @see https://nx.dev/reference/nx-json#inputs-&-namedinputs
     */
    interface NamedInputs {
        readonly [name: string]: string[];
    }
    /**
     * Target defaults config
     * @see https://nx.dev/reference/nx-json#target-defaults
     */
    interface TargetDefaults {
        readonly [name: string]: ProjectTarget;
    }
    /**
     * NX workspace configurations.
     *
     * @see https://nx.dev/configuration/packagejson
     */
    interface WorkspaceConfig {
        /**
         * Affected branch.
         *
         * @default mainline
         */
        readonly affectedBranch?: string;
        /**
         * Configuration for Implicit Dependencies.
         *
         * @see https://nx.dev/configuration/packagejson#implicitdependencies
         */
        readonly implicitDependencies?: ImplicitDependencies;
        /**
         * Configuration for TargetDependencies.
         *
         * @see https://nx.dev/configuration/packagejson#target-dependencies
         */
        readonly targetDependencies?: TargetDependencies;
        /**
         * List of patterns to include in the .nxignore file.
         *
         * @see https://nx.dev/configuration/packagejson#nxignore
         */
        readonly nxIgnore?: string[];
        /**
         * Read only access token if enabling nx cloud.
         */
        readonly nxCloudReadOnlyAccessToken?: string;
        /**
         * Defines the list of targets/operations that are cached by Nx
         *
         * @default ["build", "test"]
         * @see https://nx.dev/reference/nx-json#tasks-runner-options
         */
        readonly cacheableOperations?: string[];
        /**
         * Named inputs
         * @see https://nx.dev/reference/nx-json#inputs-&-namedinputs
         */
        readonly namedInputs?: NamedInputs;
        /**
         * Target defaults
         *
         * @see https://nx.dev/reference/nx-json#target-defaults
         */
        readonly targetDefaults?: TargetDefaults;
    }
    interface ProjectConfig {
        /**
         * Named inputs
         * @see https://nx.dev/reference/nx-json#inputs-&-namedinputs
         */
        readonly namedInputs?: NamedInputs;
        /**
         * Targets configuration
         * @see https://nx.dev/reference/project-configuration
         */
        readonly targets?: ProjectTargets;
        /**
         * Project tag annotations
         *
         * @see https://nx.dev/reference/project-configuration#tags
         */
        readonly tags?: string[];
        /**
         * Implicit dependencies
         *
         * @see https://nx.dev/reference/project-configuration#implicitdependencies
         */
        readonly implicitDependencies?: string[];
        /**
         * Explicit list of scripts for Nx to include.
         * @see https://nx.dev/reference/project-configuration#ignoring-package.json-scripts
         */
        readonly includedScripts?: string[];
    }
}
