#!/usr/bin/env ts-node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const console = require("console");
const path = require("path");
const process = require("process");
const fs = require("fs-extra");
const cdk_1 = require("projen/lib/cdk");
// The directory where our 'package.json' lives
const MONOPACKAGE_ROOT = process.cwd();
const ROOT_PATH = findWorkspacePath();
const LIBRARIES_ROOT = path.resolve(ROOT_PATH, 'packages');
const UBER_PACKAGE_JSON_PATH = path.join(MONOPACKAGE_ROOT, 'package.json');
const EXCLUDED_PACKAGES = [];
async function main() {
    console.log(`🌴  workspace root path is: ${ROOT_PATH}`);
    const uberPackageJson = await fs.readJson(UBER_PACKAGE_JSON_PATH);
    const libraries = await findLibrariesToPackage(uberPackageJson);
    await verifyDependencies(uberPackageJson, libraries);
    await prepareSourceFiles(libraries, uberPackageJson);
}
main().then(() => process.exit(0), (err) => {
    console.error('❌ An error occurred: ', err.stack);
    process.exit(1);
});
/**
 * Find the workspace root path. Walk up the directory tree until you find nx.json
 */
function findWorkspacePath() {
    return _findRootPath(process.cwd());
    function _findRootPath(part) {
        if (part === path.resolve(part, '..')) {
            throw new Error('couldn\'t find a \'nx.json\' file when walking up the directory tree, are you in a aws-pdk project?');
        }
        if (fs.existsSync(path.resolve(part, 'nx.json'))) {
            return part;
        }
        return _findRootPath(path.resolve(part, '..'));
    }
}
async function findLibrariesToPackage(uberPackageJson) {
    console.log('🔍 Discovering libraries that need packaging...');
    const deprecatedPackages = uberPackageJson.bundle?.deprecatedPackages;
    const result = new Array();
    for (const dir of await fs.readdir(LIBRARIES_ROOT)) {
        const packageJsonPath = path.resolve(LIBRARIES_ROOT, dir, 'package.json');
        if (!fs.pathExistsSync(packageJsonPath)) {
            continue;
        }
        const packageJson = await fs.readJson(packageJsonPath);
        if (packageJson.bundle?.exclude || EXCLUDED_PACKAGES.includes(packageJson.name)) {
            console.log(`\t⚠️ Skipping (bundle excluded):    ${packageJson.name}`);
            continue;
        }
        else if (packageJson.jsii == null) {
            console.log(`\t⚠️ Skipping (not jsii-enabled):   ${packageJson.name}`);
            continue;
        }
        else if (deprecatedPackages?.some(packageName => packageName === packageJson.name)) {
            console.log(`\t⚠️ Skipping (bundle deprecated): ${packageJson.name}`);
            continue;
        }
        else if (packageJson.deprecated) {
            console.log(`\t⚠️ Skipping (deprecated):         ${packageJson.name}`);
            continue;
        }
        else if (packageJson.stability !== cdk_1.Stability.STABLE) {
            console.log(`\t⚠️ Skipping (non stable):         ${packageJson.name}`);
            continue;
        }
        result.push({
            packageJson,
            root: path.join(LIBRARIES_ROOT, dir),
            shortName: packageJson.name === 'aws-prototyping-sdk' ? packageJson.name : packageJson.name.slice('@aws-prototyping-sdk/'.length),
        });
    }
    console.log(`\tℹ️ Found ${result.length} relevant packages!`);
    return result;
}
async function verifyDependencies(packageJson, libraries) {
    console.log('🧐 Verifying dependencies are complete...');
    const toBundle = {};
    for (const library of libraries) {
        for (const depName of library.packageJson.bundleDependencies ?? library.packageJson.bundledDependencies ?? []) {
            const requiredVersion = library.packageJson.devDependencies?.[depName]
                ?? library.packageJson.dependencies?.[depName]
                ?? '*';
            if (toBundle[depName] != null && toBundle[depName] !== requiredVersion) {
                throw new Error(`Required to bundle different versions of ${depName}: ${toBundle[depName]} and ${requiredVersion}.`);
            }
            toBundle[depName] = requiredVersion;
        }
    }
    const workspacePath = path.resolve(ROOT_PATH, 'package.json');
    const workspace = await fs.readJson(workspacePath);
    const spuriousBundledDeps = new Set(packageJson.bundledDependencies ?? []);
    for (const [name, version] of Object.entries(toBundle)) {
        spuriousBundledDeps.delete(name);
        const nohoist = `${packageJson.name}/${name}`;
        if (!workspace.workspaces.nohoist?.includes(nohoist)) {
            throw new Error(`\t⚠️ Missing yarn workspace nohoist: ${nohoist}`);
        }
        if (!(packageJson.bundledDependencies?.includes(name))) {
            throw new Error(`\t⚠️ Missing bundled dependency: ${name} at ${version}`);
        }
        if (packageJson.dependencies?.[name] !== version) {
            throw new Error(`\t⚠️ Missing or incorrect dependency: ${name} at ${version}`);
        }
    }
    packageJson.bundledDependencies = packageJson.bundledDependencies?.filter((dep) => !spuriousBundledDeps.has(dep));
    if (spuriousBundledDeps.size > 0) {
        throw new Error(`\t⚠️ Spurious bundled dependencies detected. Please remove from dependencies: ${spuriousBundledDeps}`);
    }
    console.log('\t✅ Dependencies are correct!');
}
async function prepareSourceFiles(libraries, packageJson) {
    console.log('📝 Preparing source files...');
    const libRoot = resolveLibRoot(packageJson);
    // Should not remove collection directory if we're currently in it. The OS would be unhappy.
    if (libRoot !== process.cwd()) {
        await fs.remove(libRoot);
    }
    const indexStatements = new Array();
    for (const library of libraries) {
        const libDir = path.join(libRoot, library.shortName);
        const copied = await transformPackage(library, packageJson, libDir, libraries);
        if (!copied) {
            continue;
        }
        indexStatements.push(`export * as ${library.shortName.replace(/-/g, '_')} from './${library.shortName}';`);
    }
    await fs.writeFile(path.join(libRoot, 'index.ts'), indexStatements.join('\n'), { encoding: 'utf8' });
    console.log('\t🍺 Success!');
}
async function transformPackage(library, uberPackageJson, destination, allLibraries) {
    await fs.mkdirp(destination);
    await copyOrTransformFiles(library.root, destination, allLibraries, uberPackageJson);
    await fs.writeFile(path.join(destination, 'index.ts'), `export * from './src';\n`, { encoding: 'utf8' });
    const config = uberPackageJson.jsii.targets;
    await fs.writeJson(path.join(destination, '.jsiirc.json'), {
        targets: transformTargets(config, library.packageJson.jsii.targets),
    }, { spaces: 2 });
    // if libRoot is _not_ under the root of the package, generate a file at the
    // root that will refer to the one under lib/ so that users can still import
    // from "monocdk/aws-lambda".
    const relativeLibRoot = uberPackageJson.bundle?.libRoot;
    if (relativeLibRoot && relativeLibRoot !== '.') {
        await fs.writeFile(path.resolve(MONOPACKAGE_ROOT, `${library.shortName}.ts`), `export * from './${relativeLibRoot}/${library.shortName}';\n`, { encoding: 'utf8' });
    }
    return true;
}
function transformTargets(monoConfig, targets) {
    if (targets == null) {
        return targets;
    }
    const result = {};
    for (const [language, config] of Object.entries(targets)) {
        switch (language) {
            case 'dotnet':
                if (monoConfig?.dotnet != null) {
                    result[language] = {
                        namespace: config.namespace,
                    };
                }
                break;
            case 'java':
                if (monoConfig?.java != null) {
                    result[language] = {
                        package: config.package,
                    };
                }
                break;
            case 'python':
                if (monoConfig?.python != null) {
                    result[language] = {
                        module: `${monoConfig.python.module}.${config.module.replace(/^aws_prototyping_sdk\./, '')}`,
                    };
                }
                break;
            default:
                throw new Error(`Unsupported language for submodule configuration translation: ${language}`);
        }
    }
    return result;
}
async function copyOrTransformFiles(from, to, libraries, uberPackageJson) {
    const promises = (await fs.readdir(from)).map(async (name) => {
        if (shouldIgnoreFile(name)) {
            return;
        }
        if (name.endsWith('.d.ts') || name.endsWith('.js')) {
            if (await fs.pathExists(path.join(from, name.replace(/\.(d\.ts|js)$/, '.ts')))) {
                // We won't copy .d.ts and .js files with a corresponding .ts file
                return;
            }
        }
        const source = path.join(from, name);
        const destination = path.join(to, name);
        const stat = await fs.stat(source);
        if (stat.isDirectory()) {
            await fs.mkdirp(destination);
            return copyOrTransformFiles(source, destination, libraries, uberPackageJson);
        }
        if (name.endsWith(".ts")) {
            const sourceCode = fs.readFileSync(source).toString()
                .replace(/(import .* from ["'])@aws-prototyping-sdk(\/.*['"];)/g, `$1${path.relative(path.dirname(destination), path.join(LIBRARIES_ROOT, "aws-prototyping-sdk"))}$2`);
            return fs.writeFile(destination, sourceCode);
        }
        else {
            return fs.copyFile(source, destination);
        }
    });
    await Promise.all(promises);
}
const IGNORED_FILE_NAMES = new Set([
    '.eslintrc.js',
    '.gitignore',
    '.jest.config.js',
    '.jsii',
    '.env',
    'target',
    'dist',
    'lib',
    '.npmignore',
    'node_modules',
    'package.json',
    'tsconfig.json',
    'tsconfig.tsbuildinfo',
    'LICENSE',
    'NOTICE',
]);
function shouldIgnoreFile(name) {
    return IGNORED_FILE_NAMES.has(name);
}
/**
 * Resolves the directory where we're going to collect all the libraries.
 *
 * By default, this is purposely the same as the monopackage root so that our
 * two import styles resolve to the same files but it can be overridden by
 * seeting `bundle.libRoot` in the package.json of the uber package.
 *
 * @param uberPackageJson package.json contents of the uber package
 * @returns The directory where we should collect all the libraries.
 */
function resolveLibRoot(uberPackageJson) {
    return path.resolve(uberPackageJson.bundle?.libRoot ?? MONOPACKAGE_ROOT);
}
//# sourceMappingURL=data:application/json;base64,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