## 简介

`fastapi-build` 是一个强大的 CLI 工具，用于搭建 FastAPI 项目脚手架。受 Django 管理功能的启发，它允许开发者快速设置 FastAPI 应用程序的基本结构和依赖项。提供视图类，仿Django ORM风格操作，自定义错误处理等集成

## 安装
必要条件： python>=3.9
```
pip install fastapi-build
```

## 快速开始

### 创建新项目

使用 `startproject` 命令创建一个新项目

```
fastapi-build startproject myproject
也可以使用 --example-api 添加demo接口示例
fastapi-build startproject --example-api myproject
```

### 创建新应用

使用 `startapp` 命令在项目中创建一个新应用

```
cd myproject
fastapi-build startapp myapp
```

### 添加插件

使用 `add_plugin` 命令向项目添加插件

```
当前支持 插件列表 db, db[mysql], db[redis], db[es], celery
fastapi-build add_plugin plugin_name
```

#### 可用插件

- **db**: 提供所有数据库支持
- **db[mysql]**: 提供 MySQL 数据库支持
- **db[redis]**: 提供 Redis 数据库支持
- **db[es]**: 提供 Elasticsearch 支持
- **celery**: 提供 Celery 任务队列支持


## 项目结构

由 `fastapi-build` 生成的项目结构概览。

```
├── README.md
├── build
│   ├── Dockerfile
│   └── docker_build.sh
├── requirements.txt
└── src
    ├── api
    │   ├── __init__.py
    │   └── demo
    │       ├── __init__.py
    │       ├── request_schema.py
    │       ├── response_schema.py
    │       ├── urls.py
    │       └── view.py
    ├── auth
    │   ├── __init__.py
    │   ├── authenticate.py
    │   └── hashers.py
    ├── common
    │   ├── __init__.py
    │   ├── load_model.py
    │   ├── log.py
    │   └── patch
    │       ├── __init__.py
    │       └── fastapi_patch.py
    ├── config
    │   ├── __init__.py
    │   ├── dev.py
    │   └── settings.py
    ├── core
    │   ├── __init__.py
    │   ├── base_params.py
    │   ├── base_view.py
    │   ├── decorator.py
    │   └── response.py
    ├── dao
    │   ├── __init__.py
    │   ├── base.py
    │   └── sql_tools.py
    ├── db
    │   ├── backends
    │   │   ├── __init__.py
    │   │   ├── es.py
    │   │   ├── mysql.py
    │   │   └── redis_client.py
    │   └── models
    │       ├── __init__.py
    │       └── base.py
    ├── exceptions
    │   ├── __init__.py
    │   ├── base.py
    │   ├── custom_exception.py
    │   ├── error_code.py
    │   └── http_status.py
    ├── gunicorn_conf.py
    ├── middleware
    │   ├── __init__.py
    │   └── register.py
    └── server.py
```

## 运行项目
```
cd src
python server.py
INFO:     Started server process [70055]
INFO:     Waiting for application startup.
INFO:     Application startup complete.
INFO:     Uvicorn running on http://127.0.0.1:6100 (Press CTRL+C to quit)


```
访问 http://127.0.0.1:6100/docs 查看文档

## 配置文件
提供了基础的数据库，时区，日志路径等配置,本地开发是dev.py 可覆盖配置
- src/config/settings.py

### 日志配置
**使用loguru管理日志**
- src/common.log.py


## 基于类的视图
1. 创建app: fbuild startapp example_api 
2. 编辑 api/example_api/urls.py
```python
from . import APP_NAME
from core.base_view import path
from .example_view import ExampleView


urlpatterns = [
    path('/test', ExampleView, tags=[APP_NAME])
]
```

3. 编写视图类
api/example_api/view.py
目前支持在 视图类中复写以下函数, 所有视图函数默认第一个参数为request对象
 - get, get请求，get查询,不带路径id
 - detail, get请求,根据路径id查询 
 - post, post请求，表单提交
 - query_post, post请求，用于复杂参数的post方式查询
 - put, put请求，根据路径id的表单更新
 - multi_put, put请求，带请求体批量更新
 - delete, delete请求, 根据路径id的删除请求
 - multi_delete, delete请求，带请求体的批量删除
```python
from typing import Annotated

from fastapi import Request, Query

from core.base_view import BaseView
from core.decorator import api_description
from core.response import Res
from .response_schema import ExampleResponseModel
from .request_schema import ExampleRequestModel


class ExampleView(BaseView):

    @api_description(summary='example view get query', response_model=Res(ExampleResponseModel))
    def get(self, request: Request, name: Annotated[str, Query(description='名称', min_length=1)]):
        return self.message(data={"name": name, "creator_id": 1})
    
    @api_description(summary='example view detail query', response_model=Res(ExampleResponseModel))
    async def detail(self, request: Request, _id: int):
        return self.message(data={"name": 'hello', "creator_id": _id})

    async def post(self, request: Request, data: ExampleRequestModel) -> Res(ExampleResponseModel):
        return self.message(data={"name": data.name, "creator_id": data.creator_id}
```

4. 启动项目
```shell
cd src
python server.py
```
5. 访问接口文档
- 根据app名称分组，接口名称及参数可添加注释，
- 返回结构为 {"code": 0, "data": [], "message": ""} 制作完成
![apidocs2](./docs/asset/img/api_docs2.png)
![apidocs](./docs/asset/img/apidocs.png)


## 仿Django ORM 操作

### 类似 Django 的 ORM

提供的类似 Django 的 ORM 操作的有限集成，例如:

```python
from db.models.base import BaseModel
class User(BaseModel):
    __tablename__ = 'user'
    # you column ...
    
User.objects.get()
await User.objects.aget()

User.objects.create()
await User.objects.a_create()

User.objects.update_by_id()
await User.objects.a_update_by_id()
# ...等其他操作
```

## 中间件
内置了接口信息打印，访问时长记录，跨域CORS，自定义错误返回结构等。
- src/middleware/register.py

## 错误处理
```python
from exceptions.custom_exception import ParamsError
from exceptions.http_status import HTTP_500_INTERNAL_SERVER_ERROR
# 接口中直接使用
async def post(request: Request):
    raise ParamsError(message="username must be string")

# 将返回：
# http_code: 400
res: {"code": 400, "message": "username must be string"}

# 可指定 http_code
ParamsError(message="username must be string", http_code=HTTP_500_INTERNAL_SERVER_ERROR)

# 可自定义错误类
from exceptions.base import ApiError
class ValidatePhoneError(ApiError):
    default_code = ParamCheckError
    default_message = "请输出正确的手机号"
    default_http_code = HTTP_400_BAD_REQUEST

```

## 贡献指南

感谢你对 `fastapi-build` 的贡献！请遵循以下步骤提交你的代码：

1. Fork 此仓库
2. 创建你的特性分支 (`git checkout -b feature/fooBar`)
3. 提交你的更改 (`git commit -am 'Add some fooBar'`)
4. 推送到分支 (`git push origin feature/fooBar`)
5. 创建一个新的 Pull Request

## 许可证

此项目使用 MIT 许可证，详情请参阅 LICENSE 文件。
