""" Technology class to describe all technology dependencies. Main use is scaling.

If a technology can use TRADICA, the class :class:`DMT.TRADICA.TechTradica` is recommended!

"""
# DMT
# Copyright (C) 2019  Markus Müller and Mario Krattenmacher and the DMT contributors <https://gitlab.hrz.tu-chemnitz.de/CEDIC_Bipolar/DMT/>
#
# This file is part of DMT.
#
# DMT is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DMT is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>
from DMT.core.dut_type import DutType
try:
    from DMT.external.pylatex import Tex
    from pylatex import Section
except ImportError:
    pass

class Technology(object):
    r""" This abstract class shall bundle all relevant scaling and extraction routines that are specific for a given technologe.

    Parameters
    ----------
    name : str
        Name of the technology

    Attributes
    ----------
    name : str
        Name of the technology

    """
    name = ''
    def __init__(self, name, scaling_builder=None):
        self.name = name
        self.scaling_builder = scaling_builder

    @classmethod
    def print_tex(cls, dut_ref, mcard):
        """ Prints a technology description, mainly used for autodocumentation reasons.

        Parameters
        -----------
        dut_ref : :class:`~DMT.core.DutView`
            Can be used to obtain tech quanties... (or to generate a TRADICA input file :) )
        mcard : :class:`~DMT.core.McParameterComposition`
            A Modelcard that contains all parameters that are required for scaling, as well as the parameters that shall be scaled.
        """
        doc = Tex()
        with doc.create(Section('Technology :' + cls.name)):
            doc.append('Technology description missing')
        return doc

    @classmethod
    def get_bib_entries(cls):
        """ bibliograpy entries of a technology
        """
        return ""

    @classmethod
    def create_mcard_library(cls, lib, mcard, path, dut_type=DutType.npn):
        """ Creates a file containing model cards for all sizes of duts present in the lib.

        Parameters
        -----------
        lib : :class:`~DMT.core.DutLib`
        path : str
            File path to write to.
        dut_type : :class:`~DMT.core.DutType`, optional
            Dut types to create model cards for.
        """

    @classmethod
    def scale_all(cls, mcard, lE0, bE0, nfinger, config):
        """ This method receives a Modelcard (that includes relevant scaling parameters such as sheet resistances) and sets the scaled values accordingly.

        Parameters
        ----------
        mcard : :class:`~DMT.core.McParameterComposition`
            A Modelcard or McParameterCompositon that contains all parameters that are required for scaling, as well as the parameters that shall be scaled.
        lE0   : float64
            The length of the desired emitter window to be modeled by mcard.
        bE0   : float64
            The width of the desired emitter window to be modeled by mcard.
        nfinger : integer
            Number of emitter fingers.
        config : str
            A unique identifier for the configuration.
        """
        mcard = cls.scale_capacitances(mcard, lE0, bE0, nfinger, config)
        mcard = cls.scale_resistances(mcard, lE0, bE0, nfinger, config)
        mcard = cls.scale_currents(mcard, lE0, bE0, nfinger, config)

        return mcard

    @classmethod
    def scale_currents(cls, mcard, lE0, bE0, nfinger, config):
        """ This method receives a Modelcard (that includes relevant scaling parameters such as sheet resistances) and sets the scaled currents accordingly.

        Parameters
        ----------
        mcard : :class:`~DMT.core.McParameterComposition`
            A Modelcard or McParameterCompositon that contains all parameters that are required for scaling, as well as the parameters that shall be scaled.
        lE0   : float64
            The length of the desired emitter window to be modeled by mcard.
        bE0   : float64
            The width of the desired emitter window to be modeled by mcard.
        nfinger : integer
            Number of emitter fingers.
        config : str
            A unique identifier for the configuration.
        """
        raise NotImplementedError

    @classmethod
    def scale_capacitances(cls, mcard, lE0, bE0, nfinger, config):
        """ This method receives a Modelcard (that includes relevant scaling parameters such as sheet resistances) and sets the scaled capacitances accordingly.

        Parameters
        ----------
        mcard : :class:`~DMT.core.McParameterComposition`
            A Modelcard or McParameterCompositon that contains all parameters that are required for scaling, as well as the parameters that shall be scaled.
        lE0   : float64
            The length of the desired emitter window to be modeled by mcard.
        bE0   : float64
            The width of the desired emitter window to be modeled by mcard.
        nfinger : integer
            Number of emitter fingers.
        config : str
            A unique identifier for the configuration.
        """
        raise NotImplementedError

    @classmethod
    def scale_sheet_resistances(cls, mcard, lE0, bE0, nfinger, config):
        """ This method receives a Modelcard (that includes relevant scaling parameters such as sheet resistances) and sets the scaled resistances accordingly.

        Parameters
        ----------
        mcard : :class:`~DMT.core.McParameterComposition`
            A Modelcard or McParameterComposition that contains all parameters that are required for scaling, as well as the parameters that shall be scaled.
        lE0   : float64
            The length of the desired emitter window to be modeled by mcard.
        bE0   : float64
            The width of the desired emitter window to be modeled by mcard.
        nfinger : integer
            Number of emitter fingers.
        config : str
            A unique identifier for the configuration.
        """
        raise NotImplementedError

    @classmethod
    def scale_modelcard(cls, mcard, lE0, bE0, nfinger, config, lE_drawn_ref=None, bE_drawn_ref=None):
        """ This method scales a already finished modelcard (no sheet resistances).

        Parameters
        ----------
        mcard : :class:`~DMT.core.McParameterComposition`
            A Modelcard or McParameterCompositon that contains all parameters that are required for scaling, as well as the parameters that shall be scaled.
        lE0   : float64
            The length of the desired emitter window to be modeled by mcard.
        bE0   : float64
            The width of the desired emitter window to be modeled by mcard.
        nfinger : integer
            Number of emitter fingers.
        config : str
            A unique identifier for the configuration.
        """
        raise NotImplementedError