""" DMT description of a circuit.

Must be used to describe a circuit and then passed to a circuit simulator dut.

Later on this can be extended to allow (pseudo-)simulations directly inside DMT.

"""
# DMT
# Copyright (C) 2019  Markus Müller and Mario Krattenmacher and the DMT contributors <https://gitlab.hrz.tu-chemnitz.de/CEDIC_Bipolar/DMT/>
#
# This file is part of DMT.
#
# DMT is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DMT is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>

from DMT.core import MCard

class Circuit(object):
    """ Circuit description as a list of :class:`CircuitElement`

    Parameters
    -----------
    circuit_elements : list(CircuitElement or str)
        Either directly the netlist elements as a list of CircuitElements or strings (for equations)

    Attributes
    -----------
    netlist : list(CircuitElement)
    """
    def __init__(self, circuit_elements):
        if isinstance(circuit_elements, str):
            raise NotImplementedError('The default circuit has been moved into the corresponding module. Access the circuit from there!')

        for i_element, element in enumerate(circuit_elements):
            if not isinstance(element, (CircuitElement, str)):
                raise TypeError("The netlist has to be a list of CircuitElement or str! At position " + str(i_element) + " is a entry of type " + type(element))

        self.netlist = circuit_elements


RESISTANCE = 'R'
""" Indicates a resistance in a model equivalent circuit. """
CAPACITANCE = 'C'
""" Indicates a capacitance in a model equivalent circuit. """
INDUCTANCE = 'L'
""" Indicates a inductance in a model equivalent circuit. """
CURRENT = 'I_Source'
""" Indicates a current source in a model equivalent circuit. """
VOLTAGE = 'V_Source'
""" Indicates a voltage source in a model equivalent circuit. """
SHORT = 'Short'
""" Indicates a voltage source in a model equivalent circuit. """

class CircuitElement(object):
    """ Class that is used to describe netlist elements such as resistors, capacitors or inductors.

    The possible CircuitElements define the DMT netlist format. Circuit simulators need to convert the DMT netlist format to their respective one.
    This class has a special emphasis on good error messages and typesetting.

    Possible element types:

    * :py:const:`RESISTANCE`
    * :py:const:`CAPACITANCE`
    * :py:const:`INDUCTANCE`
    * :py:const:`CURRENT`
    * :py:const:`VOLTAGE`
    * :py:const:`SHORT`
    * va_module -> could be a transistor

    Parameters
    -----------
    element_type : str
        Element type, for example: 'R' for resistor. possible values: 'V_Source', 'I_Source', 'R', 'C', 'L', 'Short' ,'va_module'
    name : str
        Element name, for example: 'R1' for resistor 1. Names should be unique within their netlist.
    contact_nodes : tuple(str)
        Contact nodes of the element, for example: ('n__1', 'n__2')
    parameters : list[tuple(str,str)], optional
        Parameters of the element, for example: [('R', '1k')]

    Attributes
    -----------
    element_type : str
        Element type, for example: 'R' for resistor
    name : str
        Element name, for example: 'R1' for resistor 1
    contact_nodes : iterable(str)
        Contact nodes of the element, for example: ('n__1', 'n__2')
    parameters : iterable[tuple(str,str)]
        Parameters of the element, for example: [('R', '1k')]
    method : callable, optional
        Used to build up the equivalent circuit in the model environment.
    """
    possible_types = [VOLTAGE, CURRENT, RESISTANCE, CAPACITANCE, INDUCTANCE, SHORT, 'va_module', 'pdk','"hbt_n1s"']

    def __init__(self, element_type, name, contact_nodes, parameters=None, method=None):
        if isinstance(parameters, MCard):
            CircuitElement.possible_types.append(parameters.default_module_name)

        if isinstance(element_type, str):
            if element_type in self.possible_types:
                self.element_type   = element_type

            else:
                raise IOError('DMT -> Circuit: Element Type '+ str(element_type) + ' is unknown. \n Possible types: ' + str(CircuitElement.possible_types))

        else:
            raise TypeError('DMT -> element_type: element_type needs to be a string.')

        if isinstance(name, str):
            self.name = name
        else:
            raise TypeError('The element name has to be a string! Given was ' + str(name) + ' of type ' + type(name) )

        if isinstance(contact_nodes, (tuple, list)):
            for i_node, node in enumerate(contact_nodes):
                if not isinstance(node, str):
                    raise TypeError('The element contact nodes have to be a tuple of strings! Given was a tuple of different elements. The node ' + str(i_node) + " had the type " + type(node))

            self.contact_nodes = contact_nodes

        else:
            raise TypeError('The element contact nodes have to be a tuple of strings! Given was ' + str(contact_nodes) + ' of type ' + type(contact_nodes) )


        if parameters is None:
            pass

        elif isinstance(parameters, list):
            for i_parameter, parameter in enumerate(parameters):
                if isinstance(parameter, tuple):
                    for parameter_part in parameter:
                        if not isinstance(parameter_part, str):
                            raise TypeError("The parameters have to be a list of tuple of strings! Given was a list of tuples with at least one element of " + type(parameter_part) + " in the tuple " + str(i_parameter))
                else:
                    raise TypeError("The parameters have to be a list of tuple of strings! Given was a list with at least one element of " + type(parameter) + " at position " + str(i_parameter))

        elif isinstance(parameters, MCard):
            # Allow model cards!
            pass

        else:
            raise TypeError("The parameters have to be a list of tuple of strings or a modelcard! Given was a " + type(parameters) )

        self.parameters = parameters

        self.method = method