"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("aws-cdk-lib");
const events = require("aws-cdk-lib/aws-events");
const defaults = require("@aws-solutions-constructs/core");
require("@aws-cdk/assert/jest");
const lib_1 = require("../lib");
function deployNewStack(stack) {
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    };
    return new lib_1.EventbridgeToSns(stack, 'test', props);
}
function deployStackWithNewEventBus(stack) {
    const props = {
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: { eventBusName: 'test' }
    };
    return new lib_1.EventbridgeToSns(stack, 'test-neweventbus', props);
}
test('check if the event rule has permission/policy in place in sns for it to be able to publish to the topic', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::SNS::TopicPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "SNS:Publish",
                        "SNS:RemovePermission",
                        "SNS:SetTopicAttributes",
                        "SNS:DeleteTopic",
                        "SNS:ListSubscriptionsByTopic",
                        "SNS:GetTopicAttributes",
                        "SNS:Receive",
                        "SNS:AddPermission",
                        "SNS:Subscribe"
                    ],
                    Condition: {
                        StringEquals: {
                            "AWS:SourceOwner": {
                                Ref: "AWS::AccountId"
                            }
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ]
                            ]
                        }
                    },
                    Resource: {
                        Ref: "testSnsTopic42942701"
                    },
                    Sid: "TopicOwnerOnlyAccess"
                },
                {
                    Action: [
                        "SNS:Publish",
                        "SNS:RemovePermission",
                        "SNS:SetTopicAttributes",
                        "SNS:DeleteTopic",
                        "SNS:ListSubscriptionsByTopic",
                        "SNS:GetTopicAttributes",
                        "SNS:Receive",
                        "SNS:AddPermission",
                        "SNS:Subscribe"
                    ],
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: {
                        Ref: "testSnsTopic42942701"
                    },
                    Sid: "HttpsOnly"
                },
                {
                    Action: "sns:Publish",
                    Effect: "Allow",
                    Principal: {
                        Service: "events.amazonaws.com"
                    },
                    Resource: {
                        Ref: "testSnsTopic42942701"
                    },
                    Sid: "2"
                }
            ],
            Version: "2012-10-17"
        },
        Topics: [
            {
                Ref: "testSnsTopic42942701"
            }
        ]
    });
});
test('check events rule properties', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::Events::Rule', {
        ScheduleExpression: "rate(5 minutes)",
        State: "ENABLED",
        Targets: [
            {
                Arn: {
                    Ref: "testSnsTopic42942701"
                },
                Id: {
                    "Fn::GetAtt": [
                        "testSnsTopic42942701",
                        "TopicName"
                    ]
                }
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStack(stack);
    expect(construct.eventsRule !== null);
    expect(construct.snsTopic !== null);
    expect(construct.encryptionKey !== null);
});
test('check the sns topic properties', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    expect(stack).toHaveResource('AWS::SNS::Topic', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testEncryptionKeyB55BFDBC",
                "Arn"
            ]
        }
    });
});
test('check the sns topic properties with existing KMS key', () => {
    const stack = new cdk.Stack();
    const key = defaults.buildEncryptionKey(stack, {
        description: 'my-key'
    });
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        },
        encryptionKey: key
    };
    new lib_1.EventbridgeToSns(stack, 'test-events-rule-sqs', props);
    expect(stack).toHaveResource('AWS::SNS::Topic', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        }
    });
    expect(stack).toHaveResource('AWS::KMS::Key', {
        Description: "my-key",
        EnableKeyRotation: true
    });
});
test('check eventbus property, snapshot & eventbus exists', () => {
    const stack = new cdk.Stack();
    const construct = deployStackWithNewEventBus(stack);
    expect(construct.eventsRule !== null);
    expect(construct.snsTopic !== null);
    expect(construct.encryptionKey !== null);
    expect(construct.eventBus !== null);
    // Check whether eventbus exists
    expect(stack).toHaveResource('AWS::Events::EventBus');
});
test('check exception while passing existingEventBus & eventBusProps', () => {
    const stack = new cdk.Stack();
    const props = {
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: {},
        existingEventBusInterface: new events.EventBus(stack, `test-existing-new-eventbus`, { eventBusName: 'test' })
    };
    const app = () => {
        new lib_1.EventbridgeToSns(stack, 'test-eventbridge-sns', props);
    };
    expect(app).toThrowError();
});
test('check custom event bus resource with props when deploy:true', () => {
    const stack = new cdk.Stack();
    const props = {
        eventBusProps: {
            eventBusName: 'testcustomeventbus'
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        }
    };
    new lib_1.EventbridgeToSns(stack, 'test-new-eventbridge-sns', props);
    expect(stack).toHaveResource('AWS::Events::EventBus', {
        Name: 'testcustomeventbus'
    });
});
//# sourceMappingURL=data:application/json;base64,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