"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToDynamoDB = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const core_1 = require("@aws-cdk/core");
const core_2 = require("@aws-solutions-constructs/core");
/**
 * @summary The ApiGatewayToDynamoDB class.
 */
class ApiGatewayToDynamoDB extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CloudFrontToApiGatewayToLambdaProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckProps(props);
        // Set the default props for DynamoDB table
        const dynamoTableProps = defaults.consolidateProps(defaults.DefaultTableProps, props.dynamoTableProps);
        let partitionKeyName = dynamoTableProps.partitionKey.name;
        if (props.existingTableObj) {
            partitionKeyName = core_2.getPartitionKeyNameFromTable(props.existingTableObj);
        }
        // Since we are only invoking this function with an existing Table or tableProps,
        // (not a table interface), we know that the implementation will always return
        // a Table object and we can safely cast away the optional aspect of the type.
        this.dynamoTable = defaults.buildDynamoDBTable(this, {
            existingTableObj: props.existingTableObj,
            dynamoTableProps: props.dynamoTableProps
        })[1];
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway Resource
        const apiGatewayResource = this.apiGateway.root.addResource("{" + partitionKeyName + "}");
        // Setup API Gateway Method
        // Create
        if (props.allowCreateOperation && props.allowCreateOperation === true && props.createRequestTemplate) {
            const createRequestTemplate = props.createRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActionToPolicy("dynamodb:PutItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "PutItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "POST",
                apiResource: this.apiGateway.root,
                requestTemplate: createRequestTemplate
            });
        }
        // Read
        if (props.allowReadOperation === undefined || props.allowReadOperation === true) {
            let readRequestTemplate;
            if (props.readRequestTemplate) {
                readRequestTemplate = props.readRequestTemplate;
            }
            else {
                readRequestTemplate =
                    `{ \
          "TableName": "${this.dynamoTable.tableName}", \
          "KeyConditionExpression": "${partitionKeyName} = :v1", \
          "ExpressionAttributeValues": { \
            ":v1": { \
              "S": "$input.params('${partitionKeyName}')" \
            } \
          } \
        }`;
            }
            this.addActionToPolicy("dynamodb:Query");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "Query",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "GET",
                apiResource: apiGatewayResource,
                requestTemplate: readRequestTemplate
            });
        }
        // Update
        if (props.allowUpdateOperation && props.allowUpdateOperation === true && props.updateRequestTemplate) {
            const updateRequestTemplate = props.updateRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActionToPolicy("dynamodb:UpdateItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "UpdateItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "PUT",
                apiResource: apiGatewayResource,
                requestTemplate: updateRequestTemplate
            });
        }
        // Delete
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            let deleteRequestTemplate;
            if (props.deleteRequestTemplate) {
                deleteRequestTemplate = props.deleteRequestTemplate;
            }
            else {
                deleteRequestTemplate =
                    `{ \
          "TableName": "${this.dynamoTable.tableName}", \
          "Key": { \
            "${partitionKeyName}": { \
              "S": "$input.params('${partitionKeyName}')" \
              } \
            }, \
          "ReturnValues": "ALL_OLD" \
        }`;
            }
            this.addActionToPolicy("dynamodb:DeleteItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "DeleteItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "DELETE",
                apiResource: apiGatewayResource,
                requestTemplate: deleteRequestTemplate
            });
        }
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.dynamoTable.tableArn
            ],
            actions: [`${action}`]
        }));
    }
}
exports.ApiGatewayToDynamoDB = ApiGatewayToDynamoDB;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToDynamoDB[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-dynamodb.ApiGatewayToDynamoDB", version: "1.153.1" };
//# sourceMappingURL=data:application/json;base64,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