from typing import Callable, List, Optional

import inquirer

from ts_cli.commands.init_cmd.kinds import KIND_DEFAULTS, KIND_TEMPLATES
from ts_cli.config.artifact_config import ArtifactConfig
from ts_cli.config.interactive_config import InteractiveConfig
from ts_cli.config.provider import Provider
from ts_cli.config.util import assert_is_any_namespace
from ts_cli.errors.critical_error import CriticalError


def map_args_prefix(args: dict, prefix: str) -> dict:
    return {
        "namespace": args.get(f"{prefix}_namespace"),
        "slug": args.get(f"{prefix}_slug"),
        "version": args.get(f"{prefix}_version"),
    }


def map_updates(
    argument: Optional[dict], *functions: Callable[[dict], Optional[dict]]
) -> Optional[dict]:
    value = argument
    for function in functions:
        if value is not None:
            updates = function(value)
            value = {**value, **updates} if updates is not None else None
    return value


class InitTemplateConfig(InteractiveConfig):
    def __init__(
        self,
        args,
        *,
        interactive: bool,
    ):
        super().__init__(args, interactive=interactive)
        self._type: str = "Template"
        default_template = (
            KIND_DEFAULTS[args.kind]
            if (args.kind and args.kind in KIND_DEFAULTS)
            else None
        )
        values = self._resolve(
            Provider.pipe(
                lambda: args.__dict__, lambda: {"template": default_template}
            ),
            ["kind", "template"],
            skip_confirmation=True,
        )
        self._provider = Provider(lambda: values)
        self.kind: str = self._provider.get("kind")
        self.template: str = self._provider.get("template")
        self._print_config_keys(self, ["kind", "template"], self._type)
        self.validate(["kind", "template"])

    def validate(self, requirements: List[str]):
        super().validate(requirements)
        if self.kind not in KIND_TEMPLATES:
            raise CriticalError(f"Unrecognized kind: '{self.kind}'")
        if self.template not in KIND_TEMPLATES[self.kind]:
            raise CriticalError(
                f"Invalid template '{self.template}' for kind '{self.kind}'. "
                f"Must be one of {KIND_TEMPLATES[self.kind]}"
            )

    @staticmethod
    def _interactive_get_kind(values: dict) -> Optional[dict]:
        return inquirer.prompt(
            [
                inquirer.List(
                    "kind",
                    message="Kind",
                    choices=[
                        "ids",
                        "protocol",
                        "task-script",
                        "tetraflow",
                    ],
                    default=values.get("kind"),
                )
            ]
        )

    @staticmethod
    def _interactive_get_template(values: dict) -> Optional[dict]:
        selected_kind = values.get("kind")

        if len(KIND_TEMPLATES[selected_kind]) == 1:
            return {"template": KIND_TEMPLATES[selected_kind][0]}

        default_template = None
        # Set selected template as default if it's valid for the selected kind
        current_template = values.get("template")
        if current_template and current_template in KIND_TEMPLATES[selected_kind]:
            default_template = current_template

        return inquirer.prompt(
            [
                inquirer.List(
                    "template",
                    message="Template",
                    choices=KIND_TEMPLATES[selected_kind],
                    default=default_template,
                )
            ]
        )

    def _interactive_resolve_values(
        self, values: dict, confirmation: Optional[inquirer.Confirm] = None
    ) -> Optional[dict]:
        return map_updates(
            values,
            self._interactive_get_kind,
            self._interactive_get_template,
            lambda _: inquirer.prompt(confirmation) if confirmation else {},
        )


class InitArtifactConfig(ArtifactConfig):
    def __init__(
        self,
        args,
        *,
        interactive: bool,
        artifact_type: str,
        type_pretty: str,
        has_function: bool,
        defaults: dict,
    ):
        super().__init__(args, interactive=interactive, has_function=has_function)
        self._type: str = type_pretty
        self.type = artifact_type
        self.prefix: str = artifact_type.replace("-", "_")
        args = args.__dict__
        non_interactive_provider = Provider.pipe(
            lambda: {"type": artifact_type},
            lambda: map_args_prefix(args, self.prefix),
            lambda: args,
            lambda: {
                "namespace": self._cli_config.get("org")
                and f"private-{self._cli_config.get('org')}"
            },
        )
        self.keys = ["type", "namespace", "slug", "version"]
        if has_function:
            self.keys.append("function")
        values = self._resolve(non_interactive_provider, self.keys)
        self._provider = Provider.pipe(lambda: values, lambda: defaults)

    def _get_correct_message(self, answers: dict) -> str:
        """
        :param answers:
        :return:
        """
        return f"Correct? [{self.format({'type': self.type,**answers}, add_function=self.has_function)}]"

    def _get_inquiry(self, existing_values: dict):
        """
        Returns a list of inquirer questions, using existing values as defaults
        :param existing_values:
        :return:
        """

        inquiry = [
            inquirer.Text(
                "namespace",
                message=f"{self._type} Namespace",
                default=existing_values.get("namespace"),
                validate=assert_is_any_namespace,
            ),
            inquirer.Text(
                "slug",
                message=f"{self._type} Slug",
                default=existing_values.get("slug"),
            ),
            inquirer.Text(
                "version",
                message=f"{self._type} Version",
                default=existing_values.get("version"),
            ),
        ]
        if self.has_function:
            inquiry.append(
                inquirer.Text(
                    "function",
                    message=f"{self._type} Function",
                    default=existing_values.get("function"),
                ),
            )
        return inquiry

    def _parse(self, values: dict) -> dict:
        return super()._parse({**values, "type": self.type})
