# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs-dev/0_00_core.ipynb.

# %% ../nbs-dev/0_00_core.ipynb 2
from __future__ import annotations
from pathlib import Path
from typing import Any, List, Optional

import configparser
import numpy as np
import sys

# %% auto 0
__all__ = ['is_type', 'validate_path', 'safe_path', 'get_config_value', 'IsLocalMachine', 'files_in_tree', 'nbs_root_dir']

# %% ../nbs-dev/0_00_core.ipynb 5
def is_type(
    obj:Any,                 # object whose type to validate
    obj_type:type,                # expected type for `obj`
    raise_error:bool=False,  # when True, raise a ValueError is `obj` is not of the right type
)-> bool:                    # True when `obj` is of the right type, False otherwise 
    """Validate that `obj` is of type `obj_type`. Raise error in the negative when `raise_error` is `True`"""
    if not isinstance(obj_type, type): raise ValueError(f"{obj_type} is not a type")
    if isinstance(obj, obj_type): return True
    else:
        if raise_error: raise ValueError(f"passed object is not of type {obj_type}")
        else: return False

# %% ../nbs-dev/0_00_core.ipynb 9
def validate_path(
    path:str|Path,           # path to validate
    path_type:str='file',    # type of the target path: `'file'`, `'dir'` or `'any'`
    raise_error:bool=False,  # when True, raise a ValueError is path does not a file
)-> bool:                    # True when path is a valid path, False otherwise 
    """Validate that path is a Path or str and points to a real file or directory"""
    if isinstance(path, str): 
        path = Path(path)
    if (path_type=='file' and path.is_file()) or (path_type=='dir' and path.is_dir()) :
        return True
    if path_type=='any' and path.exists():
        return True
    else:
        if raise_error: raise ValueError(f"No file at {path.absolute()}. Check the path")
        else: return False

# %% ../nbs-dev/0_00_core.ipynb 16
def safe_path(
    path:str|Path, # path to validate
)-> Path:          # validated path as a  pathlib.Path
    """Return a Path object when given a valid path as a string or a Path, raise error otherwise"""
    validate_path(path, path_type='any', raise_error=True)
    if isinstance(path, str): 
        path = Path(path)
    return path

# %% ../nbs-dev/0_00_core.ipynb 19
def get_config_value(section:str,                        # section in the configparser cfg file
                     key:str,                            # key in the selected section
                     path_to_config_file:Path|str=None   # path to the cfg file
                    )-> Any :                            # the value corresponding to section>key>value 
    """Returns the value corresponding to the key-value pair in the configuration file (configparser format)"""
    # validate path_to_config_file
    if path_to_config_file is None:
        path_to_config_file = Path('/content/gdrive/MyDrive/private-across-accounts/config-api-keys.cfg')
    safe_path(path_to_config_file)

    configuration = configparser.ConfigParser()
    configuration.read(path_to_config_file)
    return configuration[section][key]

# %% ../nbs-dev/0_00_core.ipynb 26
class IsLocalMachine:
    """Callable singleton class to identify if current machine was registered as local machine or not"""
    
    _instance = None
    _config_dir = '.ecutilities'
    _config_fname = 'ecutilities.cfg'
    
    def __new__(cls):
        if cls._instance is None:
            cls.home = Path.home().absolute()
            cls._instance = super().__new__(cls)
        return cls._instance
    
    @property
    def os(self): return sys.platform
    
    @property
    def home(self): return Path.hone().absolute()
    
    @property
    def p2config(self): return self.home / self._config_dir / self._config_fname
    
    def __call__(self): return self.is_local
    
    def read_config(self):
        """Read config from the configuration file if it exists and return an empty config in does not"""
        cfg = configparser.ConfigParser()
        if self.p2config.is_file(): 
            cfg.read(self.p2config)
        else:
            cfg.add_section('Infra')
        return cfg
        
    @property
    def is_local(self):
        """Return `True` if the current machine was registered as a local machine"""
        cfg = self.read_config()
        return cfg['Infra'].getboolean('registered_as_local', False)
    
    def register_as_local(self):
        """Update the configuration file to register the machine as local machine"""
        cfg = self.read_config()
        os.makedirs(self.home/self._config_dir, exist_ok=True)
        cfg['Infra']['registered_as_local'] = 'True'
        with open(self.p2config, 'w') as fp:
            cfg.write(fp)
        return cfg

# %% ../nbs-dev/0_00_core.ipynb 37
def files_in_tree(
    path: str|Path,               # path to the directory to scan  
    pattern: str|None = None      # pattern (glob style) to match in file name to filter the content
):
    """List files in directory and its subdiretories, print tree starting from parent directory"""
    validate_path(path, path_type='dir', raise_error=True)

    pattern = '*' if pattern is None else f"*{pattern}*"
    parents = [p.name for p in path.parents]
    paths = []
    pad = ' ' * 2
    idx = 0
    print(f"{parents[0]}")
    print(f"{pad}|--{path.name}")
    for f in [p for p in path.glob(pattern) if p.is_file()]:
        paths.append(f)
        print(f"{pad}|{pad*2}|--{f.name} ({idx})")
        idx += 1
    for d in [p for p in path.iterdir() if p.is_dir()]:
        print(f"{pad}|{pad*2}|--{d.name}")
        for f in [p for p in d.glob(pattern) if p.is_file()]:
            paths.append(f)
            print(f"{pad}|{pad*2}|{pad*2}|--{f.name} ({idx})")
            idx += 1
    return paths

# %% ../nbs-dev/0_00_core.ipynb 41
def nbs_root_dir(
    path:str|Path|None = None, # path from where to seek for notebook parent directory
    pattern:str = 'nbs',       # pattern to identify the nbs directory
)-> Path:                      # path of the parent directory
    """Climb directory tree up to directory including pattern ('nbs'), and return its path"""
    if path is None: path = Path()
    path = safe_path(path).absolute()
    tree = [path.name] + [p.name for p in path.parents]
    mask = [True if n.startswith(pattern) else False for n in tree]
    tree = tree[mask.index(True):]
    tree.reverse()
    nbs = Path('/'.join(tree))
    return nbs
