import { aws_ec2, aws_iam, aws_kms, aws_secretsmanager, aws_rds, Duration, RemovalPolicy, Resource } from "aws-cdk-lib";
import { Construct } from "constructs";
/**
 *  Properties to configure an Aurora Serverless Cluster
 *
 */
export interface ServerlessClusterFromSnapshotProps {
    /**
     * What kind of database to start
     */
    readonly engine: aws_rds.IClusterEngine;
    /**
     * Credentials for the administrative user
     *
     * @default - A username of 'admin' and SecretsManager-generated password
     */
    readonly credentials?: aws_rds.Credentials;
    /**
     * An optional identifier for the cluster
     *
     * @default - A name is automatically generated.
     */
    readonly clusterIdentifier?: string;
    /**
     * The identifier for the DB snapshot or DB cluster snapshot to restore from.
     *
     * You can use either the name or the Amazon Resource Name (ARN) to specify a DB cluster snapshot. However, you can use only the ARN to specify a DB snapshot.
     *
     * After you restore a DB cluster with a SnapshotIdentifier property, you must specify the same SnapshotIdentifier property for any future updates to the DB cluster. When you specify this property for an update, the DB cluster is not restored from the snapshot again, and the data in the database is not changed. However, if you don't specify the SnapshotIdentifier property, an empty DB cluster is created, and the original DB cluster is deleted. If you specify a property that is different from the previous snapshot restore property, a new DB cluster is restored from the specified SnapshotIdentifier property, and the original DB cluster is deleted.
     */
    readonly snapshotIdentifier: string;
    /**
     * The number of days during which automatic DB snapshots are retained.
     * Automatic backup retention cannot be disabled on serverless clusters.
     * Must be a value from 1 day to 35 days.
     *
     * @default Duration.days(1)
     */
    readonly backupRetention?: Duration;
    /**
     * Name of a database which is automatically created inside the cluster
     *
     * @default - Database is not created in cluster.
     */
    readonly defaultDatabaseName?: string;
    /**
     * Indicates whether the DB cluster should have deletion protection enabled.
     *
     * @default - true if removalPolicy is RETAIN, false otherwise
     */
    readonly deletionProtection?: boolean;
    /**
     * Whether to enable the Data API.
     *
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/data-api.html
     *
     * @default false
     */
    readonly enableDataApi?: boolean;
    /**
     * The VPC that this Aurora Serverless cluster has been created in.
     */
    readonly vpc: aws_ec2.IVpc;
    /**
     * Where to place the instances within the VPC
     *
     * @default - the VPC default strategy if not specified.
     */
    readonly vpcSubnets?: aws_ec2.SubnetSelection;
    /**
     * Scaling configuration of an Aurora Serverless database cluster.
     *
     * @default - Serverless cluster is automatically paused after 5 minutes of being idle.
     *   minimum capacity: 2 ACU
     *   maximum capacity: 16 ACU
     */
    readonly scaling?: aws_rds.ServerlessScalingOptions;
    /**
     * The removal policy to apply when the cluster and its instances are removed
     * from the stack or replaced during an update.
     *
     * @default - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Security group.
     *
     * @default - a new security group is created.
     */
    readonly securityGroups?: aws_ec2.ISecurityGroup[];
    /**
     * The KMS key for storage encryption.
     *
     * If you specify the SnapshotIdentifier property, the StorageEncrypted property value is inherited from the snapshot, and if the DB cluster is encrypted, the specified KmsKeyId property is used.
     *
     * @default - the default master key will be used for storage encryption
     */
    readonly storageEncryptionKey?: aws_kms.IKey;
    /**
     * Additional parameters to pass to the database engine
     *
     * @default - no parameter group.
     */
    readonly parameterGroup?: aws_rds.IParameterGroup;
    /**
     * Existing subnet group for the cluster.
     *
     * @default - a new subnet group will be created.
     */
    readonly subnetGroup?: aws_rds.ISubnetGroup;
}
/**
 * A Serverless Cluster restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
export declare class ServerlessClusterFromSnapshot extends Resource implements aws_rds.IServerlessCluster {
    /**
     * Identifier of the cluster
     */
    readonly clusterIdentifier: string;
    /**
     * The endpoint to use for read/write operations
     */
    readonly clusterEndpoint: aws_rds.Endpoint;
    /**
     * The endpoint to use for read/write operations
     */
    readonly clusterReadEndpoint: aws_rds.Endpoint;
    /**
     * Access to the network connections
     */
    readonly connections: aws_ec2.Connections;
    /**
     * The secret attached to this cluster
     */
    readonly secret?: aws_secretsmanager.ISecret;
    protected enableDataApi?: boolean;
    private readonly subnetGroup;
    private readonly vpc;
    private readonly vpcSubnets?;
    private readonly singleUserRotationApplication;
    private readonly multiUserRotationApplication;
    constructor(scope: Construct, id: string, props: ServerlessClusterFromSnapshotProps);
    /**
     * Adds the single user rotation of the master password to this cluster.
     */
    addRotationSingleUser(options?: aws_rds.RotationSingleUserOptions): aws_secretsmanager.SecretRotation;
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id: string, options: aws_rds.RotationMultiUserOptions): aws_secretsmanager.SecretRotation;
    private renderScalingConfiguration;
    /**
     * The ARN of the cluster
     */
    get clusterArn(): string;
    /**
     * Grant the given identity to access to the Data API, including read access to the secret attached to the cluster if present
     *
     * @param grantee The principal to grant access to
     */
    grantDataApiAccess(grantee: aws_iam.IGrantable): aws_iam.Grant;
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget(): aws_secretsmanager.SecretAttachmentTargetProps;
}
