"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../core/lib");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_2 = require("../lib");
const asset_constants_1 = require("./asset-constants");
let app;
let stack;
let wfstack;
let vpc;
let renderQueue;
let rcsImage;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'infraStack', {
        env: {
            region: 'us-east-1',
        },
    });
    vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
    const version = new lib_2.VersionQuery(stack, 'VersionQuery');
    renderQueue = new lib_2.RenderQueue(stack, 'RQ', {
        vpc,
        images: { remoteConnectionServer: rcsImage },
        repository: new lib_2.Repository(stack, 'Repository', {
            vpc,
            version,
        }),
        version,
    });
    wfstack = new core_1.Stack(app, 'workerFleetStack', {
        env: {
            region: 'us-east-1',
        },
    });
});
test('default worker fleet is created correctly', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    // THEN
    assert_1.expect(wfstack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.large',
        spotPrice: assert_1.ABSENT,
    }));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        ToPort: parseInt(renderQueue.endpoint.portAsString(), 10),
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                stack.getLogicalId(fleet.fleet.connections.securityGroups[0].node.defaultChild),
                'GroupId',
            ],
        },
        GroupId: {
            'Fn::ImportValue': 'infraStack:ExportsOutputFnGetAttRQLBSecurityGroupAC643AEDGroupId8F9F7830',
        },
    }));
    assert_1.expect(wfstack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/workerFleet',
    }));
    expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
    expect(fleet.node.metadata[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
    expect(fleet.node.metadata[1].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
    expect(fleet.node.metadata[1].data).toContain('being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy');
});
test('security group is added to fleet after its creation', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    fleet.addSecurityGroup(aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
        allowAllOutbound: false,
    }));
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SecurityGroups: [
            {
                'Fn::GetAtt': [
                    stack.getLogicalId(fleet.fleet.connections.securityGroups[0].node.defaultChild),
                    'GroupId',
                ],
            },
            'sg-123456789',
        ],
    }));
});
test('WorkerFleet uses given security group', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
        securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
            allowAllOutbound: false,
        }),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SecurityGroups: [
            'sg-123456789',
        ],
    }));
});
test('default worker fleet is created correctly with linux image', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration'));
});
test('default worker fleet is created correctly with spot config', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        spotPrice: 2.5,
    });
    // THEN
    assert_1.expect(wfstack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SpotPrice: '2.5',
    }));
});
test('default worker fleet is not created with incorrect spot config', () => {
    // WHEN
    expect(() => {
        new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            spotPrice: lib_2.WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT + 1,
        });
    }).toThrowError(/Invalid value: 256 for property 'spotPrice'. Valid values can be any decimal between 0.001 and 255./);
    // WHEN
    expect(() => {
        new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet2', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            spotPrice: lib_2.WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT / 2,
        });
    }).toThrowError(/Invalid value: 0.0005 for property 'spotPrice'. Valid values can be any decimal between 0.001 and 255./);
});
test('default worker fleet is created correctly custom Instance type', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.medium',
    }));
});
test.each([
    'test-prefix/',
    '',
])('default worker fleet is created correctly with custom LogGroup prefix %s', (testPrefix) => {
    // GIVEN
    const id = 'workerFleet';
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, id, {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        logGroupProps: {
            logGroupPrefix: testPrefix,
        },
    });
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: testPrefix + id,
    }));
});
test('default worker fleet is created correctly custom subnet values', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    const workers = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        healthCheckConfig: {
            port: 6161,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: [{
                Ref: 'VPC1AzPublicSubnet1Subnet9649CC17',
            }],
    }));
    const userData = stack.resolve(workers.fleet.userData.render());
    expect(userData).toStrictEqual({
        'Fn::Join': [
            '',
            [
                '#!/bin/bash\nfunction exitTrap(){\nexitCode=$?\n/opt/aws/bin/cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}\ntrap exitTrap EXIT\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\' \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'set -e\n' +
                    'chmod +x \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                {
                                    Ref: asset_constants_1.CWA_ASSET_LINUX.Key,
                                },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\'\n\'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\' ',
                { Ref: 'workerFleetStringParameterDB3717DA' },
                '\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                    '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                    'fi\n' +
                    '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '" --render-queue "http://',
                {
                    'Fn::GetAtt': [
                        'RQLB3B7B1CBC',
                        'DNSName',
                    ],
                },
                ':8080" \n' +
                    'rm -f "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\"\n' +
                    'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                    '  service deadline10launcher restart\n' +
                    'fi\n' +
                    "mkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                `' '' '' '' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION}'`,
            ],
        ],
    });
});
test('default worker fleet is created correctly with groups, pools and region', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    const workers = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        groups: ['A', 'B'],
        pools: ['C', 'D'],
        region: 'E',
    });
    // THEN
    const userData = stack.resolve(workers.fleet.userData.render());
    expect(userData).toStrictEqual({
        'Fn::Join': ['', [
                '#!/bin/bash\nfunction exitTrap(){\nexitCode=$?\n/opt/aws/bin/cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}\ntrap exitTrap EXIT\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' ",
                { Ref: 'workerFleetStringParameterDB3717DA' },
                '\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                    '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                    'fi\n' +
                    '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '" --render-queue "http://',
                {
                    'Fn::GetAtt': [
                        'RQLB3B7B1CBC',
                        'DNSName',
                    ],
                },
                ':8080" \n' +
                    'rm -f "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\"\n' +
                    'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                    '  service deadline10launcher restart\n' +
                    'fi\n' +
                    "mkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                `' 'a,b' 'c,d' 'E' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION}'`,
            ]],
    });
});
test('worker fleet does validation correctly with groups, pools and region', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // group name as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            groups: ['A', 'none'],
        });
    }).toThrowError();
    // group name with whitespace
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet1', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            groups: ['A', 'no ne'],
        });
    }).toThrowError(/Invalid value: no ne for property 'groups'/);
    // pool name with whitespace
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet2', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            pools: ['A', 'none'],
        });
    }).toThrowError(/Invalid value: none for property 'pools'/);
    // pool name as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet3', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            pools: ['A', 'none'],
        });
    }).toThrowError(/Invalid value: none for property 'pools'/);
    // region as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet4', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none',
        });
    }).toThrowError(/Invalid value: none for property 'region'/);
    // region as 'all'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet5', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'all',
        });
    }).toThrowError(/Invalid value: all for property 'region'/);
    // region as 'unrecognized'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet6', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'unrecognized',
        });
    }).toThrowError(/Invalid value: unrecognized for property 'region'/);
    // region with invalid characters
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet7', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none@123',
        });
    }).toThrowError(/Invalid value: none@123 for property 'region'/);
    // region with reserved name as substring
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet8', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none123',
        });
    }).not.toThrowError();
    // region with case-insensitive name
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet9', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'None',
        });
    }).toThrowError(/Invalid value: None for property 'region'/);
});
describe('Block Device Tests', () => {
    let healthMonitor;
    beforeEach(() => {
        // create a health monitor so it does not trigger warnings
        healthMonitor = new lib_1.HealthMonitor(wfstack, 'healthMonitor', {
            vpc,
        });
    });
    test('Warning if no BlockDevices provided', () => {
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadata[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
    });
    test('No Warnings if Encrypted BlockDevices Provided', () => {
        const VOLUME_SIZE = 50;
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: true }),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        Encrypted: true,
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadata).toHaveLength(0);
    });
    test('Warnings if non-Encrypted BlockDevices Provided', () => {
        const VOLUME_SIZE = 50;
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: false }),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        Encrypted: false,
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadata[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the worker-fleet workerFleet is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
    });
    test('Warnings for BlockDevices without encryption specified', () => {
        const VOLUME_SIZE = 50;
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadata[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the worker-fleet workerFleet is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
    });
    test('No warnings for Ephemeral blockDeviceVolumes', () => {
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ephemeral(0),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    DeviceName: DEVICE_NAME,
                    VirtualName: 'ephemeral0',
                },
            ],
        }));
        expect(fleet.node.metadata).toHaveLength(0);
    });
    test('No warnings for Suppressed blockDeviceVolumes', () => {
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    DeviceName: DEVICE_NAME,
                },
            ],
        }));
        expect(fleet.node.metadata).toHaveLength(0);
    });
});
describe('HealthMonitor Tests', () => {
    let healthMonitor;
    beforeEach(() => {
        // create a health monitor so it does not trigger warnings
        healthMonitor = new lib_1.HealthMonitor(wfstack, 'healthMonitor', {
            vpc,
        });
    });
    test('Monitor is configured for Windows', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        // Ensure the configuration script is executed with the expected arguments.
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('&\'C:/temp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'63415\' \'10.1.9.2\'')));
        // Ensure that the health monitor target group has been set up.
        //  Note: It's sufficient to just check for any resource created by the HealthMonitor registration.
        //   The HealthMonitor tests cover ensuring that all of the resources are set up.
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '63415',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
    });
    test('Monitor is configured for Linux', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        // Ensure the configuration script is executed with the expected arguments.
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('\'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'63415\' \'10.1.9.2\'')));
        // Ensure that the health monitor target group has been set up.
        //  Note: It's sufficient to just check for any resource created by the HealthMonitor registration.
        //   The HealthMonitor tests cover ensuring that all of the resources are set up.
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '63415',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
    });
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'WorkerInstanceFleet',
        createConstruct: () => {
            // GIVEN
            const healthMonitorStack = new core_1.Stack(app, 'HealthMonitorStack', {
                env: {
                    region: 'us-east-1',
                },
            });
            const healthMonitor = new lib_1.HealthMonitor(healthMonitorStack, 'healthMonitor', {
                vpc,
            });
            const deviceName = '/dev/xvda';
            // WHEN
            new lib_2.WorkerInstanceFleet(wfstack, 'WorkerFleet', {
                vpc,
                workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                    'us-east-1': 'ami-any',
                }),
                renderQueue,
                healthMonitor,
                blockDevices: [{
                        deviceName,
                        volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                    }],
            });
            return wfstack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::ElasticLoadBalancingV2::TargetGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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