"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
describe('Test WorkerInstanceConfiguration for Linux', () => {
    let stack;
    let vpc;
    let instance;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
    });
    test('basic setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
        });
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '#!/bin/bash\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxDownloadRunScriptBoilerplate(asset_constants_1.CONFIG_WORKER_ASSET_LINUX),
                    `\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\'`,
                ],
            ],
        });
    });
    test('groups, pools, region setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                groups: ['g1', 'g2'],
                pools: ['p1', 'p2'],
                region: 'r1',
            },
        });
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '#!/bin/bash\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxDownloadRunScriptBoilerplate(asset_constants_1.CONFIG_WORKER_ASSET_LINUX),
                    `\' \'g1,g2\' \'p1,p2\' \'r1\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\'`,
                ],
            ],
        });
    });
    test('log setup', () => {
        // GIVEN
        const logGroupProps = {
            logGroupPrefix: '/test-prefix/',
        };
        // WHEN
        const config = new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            cloudwatchLogSettings: logGroupProps,
        });
        const ssmParam = config.node.findChild('StringParameter');
        const logGroup = config.node.findChild('ConfigLogGroupWrapper');
        const ssmParamName = stack.resolve(ssmParam.parameterName);
        const logGroupName = stack.resolve(logGroup.logGroupName);
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '#!/bin/bash\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxDownloadRunScriptBoilerplate(asset_constants_1.CWA_ASSET_LINUX),
                    '\' ',
                    ssmParamName,
                    '\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxDownloadRunScriptBoilerplate(asset_constants_1.CONFIG_WORKER_ASSET_LINUX),
                    `\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\'`,
                ],
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SSM::Parameter', {
            Value: {
                'Fn::Join': [
                    '',
                    [
                        '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"cloud-init-output-{instance_id}\",\"file_path\":\"/var/log/cloud-init-output.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"WorkerLogs-{instance_id}\",\"file_path\":\"/var/log/Thinkbox/Deadline10/deadlineslave*.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"LauncherLogs-{instance_id}\",\"file_path\":\"/var/log/Thinkbox/Deadline10/deadlinelauncher*.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                    ],
                ],
            },
        }));
    });
});
describe('Test WorkerInstanceConfiguration for Windows', () => {
    let stack;
    let vpc;
    let instance;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
    });
    test('basic setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
        });
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsDownloadRunScriptBoilerplate(asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS),
                    `\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\'` +
                        '\nif (!$?) { Write-Error \'Failed to execute the file \"C:/temp/',
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    {
                                        Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key,
                                    },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    '\"\' -ErrorAction Stop }</powershell>',
                ],
            ],
        });
    });
    test('groups, pools, region setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                groups: ['g1', 'g2'],
                pools: ['p1', 'p2'],
                region: 'r1',
            },
        });
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsDownloadRunScriptBoilerplate(asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS),
                    `\' \'g1,g2\' \'p1,p2\' \'r1\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\'` +
                        '\nif (!$?) { Write-Error \'Failed to execute the file \"C:/temp/',
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    {
                                        Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key,
                                    },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    '\"\' -ErrorAction Stop }</powershell>',
                ],
            ],
        });
    });
    test('log setup', () => {
        // GIVEN
        const logGroupProps = {
            logGroupPrefix: '/test-prefix/',
        };
        // WHEN
        const config = new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            cloudwatchLogSettings: logGroupProps,
        });
        const ssmParam = config.node.findChild('StringParameter');
        const logGroup = config.node.findChild('ConfigLogGroupWrapper');
        const ssmParamName = stack.resolve(ssmParam.parameterName);
        const logGroupName = stack.resolve(logGroup.logGroupName);
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsDownloadRunScriptBoilerplate(asset_constants_1.CWA_ASSET_WINDOWS),
                    '\' ',
                    ssmParamName,
                    '\nif (!$?) { Write-Error \'Failed to execute the file \"C:/temp/',
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    {
                                        Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key,
                                    },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    '\"\' -ErrorAction Stop }' +
                        '\nmkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsDownloadRunScriptBoilerplate(asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS),
                    `\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\'` +
                        '\nif (!$?) { Write-Error \'Failed to execute the file \"C:/temp/',
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    {
                                        Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key,
                                    },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CONFIG_WORKER_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    '\"\' -ErrorAction Stop }</powershell>',
                ],
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SSM::Parameter', {
            Value: {
                'Fn::Join': [
                    '',
                    [
                        '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"UserdataExecution-{instance_id}\",\"file_path\":\"C:\\\\ProgramData\\\\Amazon\\\\EC2-Windows\\\\Launch\\\\Log\\\\UserdataExecution.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"WorkerLogs-{instance_id}\",\"file_path\":\"C:\\\\ProgramData\\\\Thinkbox\\\\Deadline10\\\\logs\\\\deadlineslave*.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"LauncherLogs-{instance_id}\",\"file_path\":\"C:\\\\ProgramData\\\\Thinkbox\\\\Deadline10\\\\logs\\\\deadlinelauncher*.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                    ],
                ],
            },
        }));
    });
});
describe('Test WorkerInstanceConfiguration connect to RenderQueue', () => {
    let stack;
    let vpc;
    let renderQueue;
    let renderQueueSGId;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        const rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        const version = new lib_1.VersionQuery(stack, 'Version');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            version,
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
            }),
        });
        const rqSecGrp = renderQueue.connections.securityGroups[0];
        renderQueueSGId = stack.resolve(rqSecGrp.securityGroupId);
    });
    test('For Linux', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            renderQueue,
        });
        const instanceSG = instance.connections.securityGroups[0];
        const instanceSGId = stack.resolve(instanceSG.securityGroupId);
        // THEN
        // Open-box testing. We know that we invoked the connection method on the
        // render queue if the security group for the instance has an ingress rule to the RQ.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            ToPort: 8080,
            SourceSecurityGroupId: instanceSGId,
            GroupId: renderQueueSGId,
        }));
    });
    test('For Windows', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            renderQueue,
        });
        const instanceSG = instance.connections.securityGroups[0];
        const instanceSGId = stack.resolve(instanceSG.securityGroupId);
        // THEN
        // Open-box testing. We know that we invoked the connection method on the
        // render queue if the security group for the instance has an ingress rule to the RQ.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            ToPort: 8080,
            SourceSecurityGroupId: instanceSGId,
            GroupId: renderQueueSGId,
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,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