"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
test('VersionQuery constructor full version', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Stack');
    new lib_1.VersionQuery(stack, 'VersionQuery', { version: '10.1.9.2' });
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
        forceRun: assert_1.ABSENT,
        versionString: '10.1.9.2',
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: 'sts:AssumeRole',
                    Effect: 'Allow',
                    Principal: {
                        Service: 'lambda.amazonaws.com',
                    },
                },
            ],
        },
        ManagedPolicyArns: [
            {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole',
                    ],
                ],
            },
        ],
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
        Code: {
            S3Bucket: {
                Ref: asset_constants_1.VERSION_QUERY_ASSET.Bucket,
            },
            S3Key: {
                'Fn::Join': [
                    '',
                    [
                        {
                            'Fn::Select': [
                                0,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: asset_constants_1.VERSION_QUERY_ASSET.Key,
                                        },
                                    ],
                                },
                            ],
                        },
                        {
                            'Fn::Select': [
                                1,
                                {
                                    'Fn::Split': [
                                        '||',
                                        {
                                            Ref: asset_constants_1.VERSION_QUERY_ASSET.Key,
                                        },
                                    ],
                                },
                            ],
                        },
                    ],
                ],
            },
        },
        Handler: 'version-provider.handler',
        Role: {
            'Fn::GetAtt': [
                assert_1.stringLike('SingletonLambda*ServiceRole*'),
                'Arn',
            ],
        },
        Runtime: 'nodejs12.x',
    }));
});
test('VersionQuery constructor no versionString', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Stack');
    new lib_1.VersionQuery(stack, 'VersionQuery');
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
        forceRun: assert_1.stringLike('*'),
        versionString: assert_1.ABSENT,
    }));
});
test.each([
    ['10.1.9'],
    ['10.1'],
    ['10'],
])('VersionQuery constructor partial version: %s', (version) => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'Stack');
    new lib_1.VersionQuery(stack, 'VersionQuery', { version });
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
        versionString: version,
        forceRun: assert_1.stringLike('*'),
    }));
});
//# sourceMappingURL=data:application/json;base64,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