"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const env = {
    region: 'us-east-1',
};
let app;
let certificateSecret;
let versionedInstallers;
let dependencyStack;
let dockerContainer;
let images;
let lfCluster;
let licenses;
let rcsImage;
let renderQueue;
let stack;
let ubl;
let vpc;
let workerFleet;
describe('UsageBasedLicensing', () => {
    beforeEach(() => {
        // GIVEN
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DependencyStack', { env });
        versionedInstallers = new lib_1.VersionQuery(dependencyStack, 'VersionQuery');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromDockerImageAsset(new aws_ecr_assets_1.DockerImageAsset(dependencyStack, 'Image', {
            directory: __dirname,
        }));
        renderQueue = new lib_1.RenderQueue(dependencyStack, 'RQ-NonDefaultPort', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(dependencyStack, 'RepositoryNonDefault', {
                vpc,
                version: versionedInstallers,
            }),
            version: versionedInstallers,
        });
        lfCluster = new aws_ecs_1.Cluster(dependencyStack, 'licenseForwarderCluster', {
            vpc,
        });
        certificateSecret = aws_secretsmanager_1.Secret.fromSecretArn(lfCluster, 'CertSecret', 'arn:aws:secretsmanager:us-west-2:675872700355:secret:CertSecret-j1kiFz');
        dockerContainer = new aws_ecr_assets_1.DockerImageAsset(lfCluster, 'license-forwarder', {
            directory: __dirname,
        });
        images = {
            licenseForwarder: aws_ecs_1.ContainerImage.fromDockerImageAsset(dockerContainer),
        };
        const workerStack = new core_1.Stack(app, 'WorkerStack', { env });
        workerFleet = new lib_1.WorkerInstanceFleet(workerStack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(dependencyStack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        licenses = [lib_1.UsageBasedLicense.forMaya()];
        stack = new core_1.Stack(app, 'Stack', { env });
        // WHEN
        ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc,
        });
    });
    test('creates an ECS cluster', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Cluster'));
    });
    describe('creates an ASG', () => {
        test('defaults', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '1',
                MaxSize: '1',
                VPCZoneIdentifier: [
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet1Subnet*`),
                    },
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet2Subnet*`),
                    },
                ],
            }));
        });
        test('capacity can be specified', () => {
            // WHEN
            const isolatedStack = new core_1.Stack(app, 'MyStack', { env });
            new lib_1.UsageBasedLicensing(isolatedStack, 'licenseForwarder', {
                certificateSecret,
                desiredCount: 2,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '2',
                MaxSize: '2',
            }));
        });
        test('gives write access to log group', () => {
            // GIVEN
            const logGroup = ubl.node.findChild('UBLLogGroup');
            const asgRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.asg.role.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: assert_1.arrayWith('logs:CreateLogStream', 'logs:PutLogEvents'),
                        Effect: 'Allow',
                        Resource: stack.resolve(logGroup.logGroupArn),
                    }),
                    Version: '2012-10-17',
                },
                Roles: assert_1.arrayWith({ Ref: asgRoleLogicalId }),
            }));
        });
    });
    describe('creates an ECS service', () => {
        test('associated with the cluster', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                Cluster: { Ref: stack.getLogicalId(ubl.cluster.node.defaultChild) },
            }));
        });
        describe('DesiredCount', () => {
            test('defaults to 1', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: 1,
                }));
            });
            test('can be specified', () => {
                // GIVEN
                const desiredCount = 2;
                const isolatedStack = new core_1.Stack(app, 'IsolatedStack', { env });
                // WHEN
                new lib_1.UsageBasedLicensing(isolatedStack, 'UBL', {
                    certificateSecret,
                    images,
                    licenses,
                    renderQueue,
                    vpc,
                    desiredCount,
                });
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: desiredCount,
                }));
            });
        });
        test('sets launch type to EC2', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                LaunchType: 'EC2',
            }));
        });
        test('sets distinct instance placement constraint', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                PlacementConstraints: assert_1.arrayWith({ Type: 'distinctInstance' }),
            }));
        });
        test('uses the task definition', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                TaskDefinition: { Ref: stack.getLogicalId(ubl.service.taskDefinition.node.defaultChild) },
            }));
        });
        test('with the correct deployment configuration', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                DeploymentConfiguration: {
                    MaximumPercent: 100,
                    MinimumHealthyPercent: 0,
                },
            }));
        });
    });
    describe('creates a task definition', () => {
        test('container name is LicenseForwarderContainer', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Name: 'LicenseForwarderContainer',
                    },
                ],
            }));
        });
        test('container is marked essential', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                    },
                ],
            }));
        });
        test('with increased ulimits', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Ulimits: [
                            {
                                HardLimit: 200000,
                                Name: 'nofile',
                                SoftLimit: 200000,
                            },
                            {
                                HardLimit: 64000,
                                Name: 'nproc',
                                SoftLimit: 64000,
                            },
                        ],
                    },
                ],
            }));
        });
        test('with awslogs log driver', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {},
                                'awslogs-stream-prefix': 'LicenseForwarder',
                                'awslogs-region': env.region,
                            },
                        },
                    },
                ],
            }));
        });
        test('configures UBL certificates', () => {
            // GIVEN
            const taskRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_CERTIFICATES_URI',
                            Value: certificateSecret.secretArn,
                        }),
                    },
                ],
                TaskRoleArn: {
                    'Fn::GetAtt': [
                        taskRoleLogicalId,
                        'Arn',
                    ],
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: certificateSecret.secretArn,
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [
                    { Ref: core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild) },
                ],
            }));
        });
        test('uses host networking', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                NetworkMode: 'host',
            }));
        });
        test('is marked EC2 compatible only', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                RequiresCompatibilities: ['EC2'],
            }));
        });
    });
    test('License Forwarder subnet selection', () => {
        // GIVEN
        const publicSubnetIds = ['PublicSubnet1', 'PublicSubnet2'];
        const vpcFromAttributes = aws_ec2_1.Vpc.fromVpcAttributes(dependencyStack, 'AttrVpc', {
            availabilityZones: ['us-east-1a', 'us-east-1b'],
            vpcId: 'vpcid',
            publicSubnetIds,
        });
        stack = new core_1.Stack(app, 'IsolatedStack', { env });
        // WHEN
        new lib_1.UsageBasedLicensing(stack, 'licenseForwarder', {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc: vpcFromAttributes,
            vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PUBLIC },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: publicSubnetIds,
        }));
    });
    test.each([
        'test-prefix/',
        '',
    ])('License Forwarder is created with correct LogGroup prefix %s', (testPrefix) => {
        // GIVEN
        stack = new core_1.Stack(app, 'IsolatedStack', { env });
        const id = 'licenseForwarder';
        // WHEN
        new lib_1.UsageBasedLicensing(stack, id, {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc,
            logGroupProps: {
                logGroupPrefix: testPrefix,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: testPrefix + id,
        }));
    });
    describe('license limits', () => {
        test('multiple licenses with limits', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack', { env });
            // WHEN
            new lib_1.UsageBasedLicensing(isolatedStack, 'licenseForwarder', {
                vpc,
                images,
                certificateSecret,
                renderQueue,
                licenses: [
                    lib_1.UsageBasedLicense.forMaya(10),
                    lib_1.UsageBasedLicense.forVray(10),
                ],
            });
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_LIMITS',
                            Value: 'maya:10;vray:10',
                        }),
                    },
                ],
            }));
        });
        test.each([
            ['3dsMax', lib_1.UsageBasedLicense.for3dsMax(10), [27002]],
            ['Arnold', lib_1.UsageBasedLicense.forArnold(10), [5056, 7056]],
            ['Cinema4D', lib_1.UsageBasedLicense.forCinema4D(10), [5057, 7057]],
            ['Clarisse', lib_1.UsageBasedLicense.forClarisse(10), [40500]],
            ['Houdini', lib_1.UsageBasedLicense.forHoudini(10), [1715]],
            ['Katana', lib_1.UsageBasedLicense.forKatana(10), [4151, 6101]],
            ['KeyShot', lib_1.UsageBasedLicense.forKeyShot(10), [27003, 2703]],
            ['Krakatoa', lib_1.UsageBasedLicense.forKrakatoa(10), [27000, 2700]],
            ['Mantra', lib_1.UsageBasedLicense.forMantra(10), [1716]],
            ['Maxwell', lib_1.UsageBasedLicense.forMaxwell(10), [5555, 7055]],
            ['Maya', lib_1.UsageBasedLicense.forMaya(10), [27002, 2702]],
            ['Nuke', lib_1.UsageBasedLicense.forNuke(10), [4101, 6101]],
            ['RealFlow', lib_1.UsageBasedLicense.forRealFlow(10), [5055, 7055]],
            ['RedShift', lib_1.UsageBasedLicense.forRedShift(10), [5054, 7054]],
            ['Vray', lib_1.UsageBasedLicense.forVray(10), [30306]],
            ['Yeti', lib_1.UsageBasedLicense.forYeti(10), [5053, 7053]],
        ])('Test open port for license type %s', (_licenseName, license, ports) => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack', { env });
            // WHEN
            ubl = new lib_1.UsageBasedLicensing(isolatedStack, 'licenseForwarder', {
                vpc,
                certificateSecret,
                licenses: [
                    license,
                ],
                renderQueue,
                images,
            });
            ubl.grantPortAccess(workerFleet, [license]);
            // THEN
            ports.forEach(port => {
                const ublAsgSecurityGroup = ubl.asg.connections.securityGroups[0].node.defaultChild;
                const ublAsgSecurityGroupLogicalId = isolatedStack.getLogicalId(ublAsgSecurityGroup);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: port,
                    GroupId: {
                        'Fn::GetAtt': [
                            ublAsgSecurityGroupLogicalId,
                            'GroupId',
                        ],
                    },
                    SourceSecurityGroupId: 'sg-123456789',
                }));
            });
        });
        test('requires one usage based license', () => {
            // Without any licenses
            expect(() => {
                new lib_1.UsageBasedLicensing(dependencyStack, 'licenseForwarder', {
                    vpc,
                    images,
                    certificateSecret: certificateSecret,
                    licenses: [],
                    renderQueue,
                });
            }).toThrowError('Should be specified at least one license with defined limit.');
        });
    });
    describe('configures render queue', () => {
        test('adds ingress rule for asg', () => {
            const ublAsgSg = ubl.asg.connections.securityGroups[0].node.defaultChild;
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 8080,
                ToPort: 8080,
                GroupId: {
                    'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(renderQueue).stackName}:ExportsOutputFnGetAttRQNonDefaultPortLBSecurityGroup*`),
                },
                SourceSecurityGroupId: {
                    'Fn::GetAtt': [
                        core_1.Stack.of(ubl).getLogicalId(ublAsgSg),
                        'GroupId',
                    ],
                },
            }));
        });
        test('sets RENDER_QUEUE_URI environment variable', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'RENDER_QUEUE_URI',
                            Value: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'http://',
                                        {
                                            'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(renderQueue).stackName}:ExportsOutputFnGetAttRQNonDefaultPortLB*`),
                                        },
                                        ':8080',
                                    ],
                                ],
                            },
                        }),
                    },
                ],
            }));
        });
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'UsageBasedLicensing',
            createConstruct: () => {
                return stack;
            },
            resourceTypeCounts: {
                'AWS::ECS::Cluster': 1,
                'AWS::EC2::SecurityGroup': 1,
                'AWS::IAM::Role': 5,
                'AWS::AutoScaling::AutoScalingGroup': 1,
                'AWS::Lambda::Function': 1,
                'AWS::SNS::Topic': 1,
                'AWS::ECS::TaskDefinition': 1,
                'AWS::ECS::Service': 1,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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