"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceConfiguration = void 0;
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const version_1 = require("./version");
/**
 * This construct can be used to configure Deadline Workers on an instance to connect to a RenderQueue, stream their log files to CloudWatch, and configure various settings of the Deadline Worker.
 *
 * The configuration happens on instance start-up using user data scripting.
 *
 * Security Considerations
 * ------------------------
 * - The instances configured by this construct will download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 */
class WorkerInstanceConfiguration extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (props.cloudwatchLogSettings) {
            this.configureCloudWatchLogStream(props.worker, id, props.cloudwatchLogSettings);
        }
        (_a = props.renderQueue) === null || _a === void 0 ? void 0 : _a.configureClientInstance({ host: props.worker });
        this.configureWorkerSettings(props.worker, id, props.workerSettings);
    }
    /**
     * This method can be used to configure a Deadline Worker instance to stream its logs to the AWS CloudWatch service.
     *
     * The logs that this configures to stream are:
     * - EC2 Instance UserData execution; this is the startup scripting that is run when the instance launches
     *    for the first time.
     * - Deadline Worker logs.
     * - Deadline Launcher logs.
     *
     * @param worker The worker to configure.
     * @param id Identifier to disambiguate the resources that are created.
     * @param logGroupProps Configuration for the log group in CloudWatch.
     */
    configureCloudWatchLogStream(worker, id, logGroupProps) {
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, `${id}LogGroupWrapper`, id, logGroupProps);
        logGroup.grantWrite(worker);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(core_1.Duration.seconds(15));
        if (worker.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'UserdataExecution', 'C:\\ProgramData\\Amazon\\EC2-Windows\\Launch\\Log\\UserdataExecution.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlineslave*.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlinelauncher*.log');
        }
        else {
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', '/var/log/Thinkbox/Deadline10/deadlineslave*.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', '/var/log/Thinkbox/Deadline10/deadlinelauncher*.log');
        }
        new core_2.CloudWatchAgent(this, `${id}LogsConfig`, {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: worker,
        });
    }
    /**
     * This method can be used to set up the Deadline Worker application on an EC2 instance.
     *
     * From a practical
     * perspective, this is executing the script found in aws-rfdk/lib/deadline/scripts/[bash,powershell]/configureWorker.[sh,ps1]
     * to configure the Deadline Worker application.
     *
     * @param worker The worker to configure.
     * @param id Identifier to disambiguate the resources that are created.
     * @param settings The Deadline Worker settings to apply.
     */
    configureWorkerSettings(worker, id, settings) {
        var _a, _b, _c, _d, _e;
        const configureWorkerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, `${id}ConfigScript`, {
            osType: worker.osType,
            baseName: 'configureWorker',
            rootDir: path.join(__dirname, '..', 'scripts/'),
        });
        // Converting to lower case, as groups and pools are all stored in lower case in deadline.
        const groups = (_b = (_a = settings === null || settings === void 0 ? void 0 : settings.groups) === null || _a === void 0 ? void 0 : _a.map(val => val.toLowerCase()).join(',')) !== null && _b !== void 0 ? _b : '';
        const pools = (_d = (_c = settings === null || settings === void 0 ? void 0 : settings.pools) === null || _c === void 0 ? void 0 : _c.map(val => val.toLowerCase()).join(',')) !== null && _d !== void 0 ? _d : '';
        configureWorkerScriptAsset.executeOn({
            host: worker,
            args: [
                `'${groups}'`,
                `'${pools}'`,
                `'${(_e = settings === null || settings === void 0 ? void 0 : settings.region) !== null && _e !== void 0 ? _e : ''}'`,
                `'${version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`,
            ],
        });
    }
}
exports.WorkerInstanceConfiguration = WorkerInstanceConfiguration;
//# sourceMappingURL=data:application/json;base64,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