"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Version = void 0;
/**
 * This class is reposonsible to do basic operations on version format.
 */
class Version {
    /**
     *
     */
    constructor(components) {
        if (components.length != 4) {
            throw new Error('Invalid version format. Version should contain exactly 4 components.');
        }
        components.forEach((component) => {
            if (component < 0) {
                throw new RangeError('Invalid version format. None of the version components can be negative.');
            }
            if (!Number.isInteger(component)) {
                throw new RangeError('Invalid version format. None of the version components can contain decimal values.');
            }
        });
        this.components = components;
    }
    /**
     * This method parses the input string and returns the version object.
     *
     * @param version version string to parse.
     */
    static parse(version) {
        if (!Version.validateVersionFormat(version)) {
            throw new TypeError(`Invalid version format. Expected format 'a.b.c.d', found '${version}'`);
        }
        return new Version(version.split('.').map(x => parseInt(x)));
    }
    /**
     * This method validates the given string for a sequence '.' separated numbers.
     *
     * @param version the string to be validated.
     *
     * @returns true if the format is correct, else false.
     */
    static validateVersionFormat(version) {
        /**
         * Regex: ^\d+(?:\.\d+){3}$
         * Matches a sequence of '.' separated numbers with exactly 4 digits.
         * - ^ asserts position at start of a line.
         * - \d+ Matches one or more digits.
         * - (?:\.\d+) Matches a dot and the following one or more digits.
         * - {3} Matches previous pattern exactly 3 times.
         * - $ asserts position at the end of a line
         */
        if (version.match(/^\d+(?:\.\d+){3}$/)) {
            return true;
        }
        return false;
    }
    /**
     * The major version number.
     *
     * @inheritdoc true
     */
    get majorVersion() {
        return this.components[0];
    }
    /**
     * The minor version number.
     *
     * @inheritdoc true
     */
    get minorVersion() {
        return this.components[1];
    }
    /**
     * The release version number.
     *
     * @inheritdoc true
     */
    get releaseVersion() {
        return this.components[2];
    }
    /**
     * The patch version number.
     *
     * @inheritdoc true
     */
    get patchVersion() {
        return this.components[3];
    }
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is greater than the provided version;
     * false if this version is less than or equal to the provided verison.
     */
    isGreaterThan(version) {
        return this.compare(version) > 0;
    }
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is less than the provided version;
     * false if this version is greater than or equal to the provided verison.
     */
    isLessThan(version) {
        return this.compare(version) < 0;
    }
    /**
     * This method compares two version strings.
     *
     * @returns true if this version is equal to the provided version;
     * false otherwise.
     */
    isEqual(version) {
        return this.compare(version) == 0;
    }
    /**
     * The method returns the version components in dot separated string format.
     */
    toString() {
        return this.components.join('.');
    }
    /**
     * This method compares 2 versions.
     *
     * @param version version to compare
     *
     * @returns negative value if this version is less than the provided version;
     * 0 if both the versions are equal;
     * positive value if this version is greater than the provided verison.
     */
    compare(version) {
        if (this.components.length != version.components.length) {
            throw new TypeError('Component count in both the versions should be same.');
        }
        for (let i = 0; i < version.components.length; i++) {
            const diff = this.components[i] - version.components[i];
            if (diff != 0) {
                return diff;
            }
        }
        return 0;
    }
}
exports.Version = Version;
/**
 * This variable holds the value for minimum supported deadline version.
 */
Version.MINIMUM_SUPPORTED_DEADLINE_VERSION = new Version([10, 1, 9, 2]);
//# sourceMappingURL=data:application/json;base64,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