"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionQuery = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
class VersionQueryBase extends core_1.Construct {
}
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 *
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 */
class VersionQuery extends VersionQueryBase {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        const lambdaCode = aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '../..', 'core', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '2e19e243-16ee-4d1a-a3c9-18d35eddd446',
            description: 'Used by the Version construct to get installer locations for a specific Deadline version.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'version-provider.handler',
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const deadlineProperties = {
            versionString: props === null || props === void 0 ? void 0 : props.version,
            // If we don't have a full static version string, create a random string that will force the Lambda to always
            // run on redeploys, effectively checking for version updates.
            forceRun: this.forceRun(props === null || props === void 0 ? void 0 : props.version),
        };
        const deadlineResource = new core_1.CustomResource(this, 'DeadlineResource', {
            serviceToken: lambdaFunc.functionArn,
            properties: deadlineProperties,
            resourceType: 'Custom::RFDK_DEADLINE_INSTALLERS',
        });
        this.majorVersion = core_1.Token.asNumber(deadlineResource.getAtt('MajorVersion'));
        this.minorVersion = core_1.Token.asNumber(deadlineResource.getAtt('MinorVersion'));
        this.releaseVersion = core_1.Token.asNumber(deadlineResource.getAtt('ReleaseVersion'));
        this.linuxInstallers = {
            patchVersion: core_1.Token.asNumber(deadlineResource.getAtt('LinuxPatchVersion')),
            repository: {
                objectKey: core_1.Token.asString(deadlineResource.getAtt('LinuxRepositoryInstaller')),
                s3Bucket: aws_s3_1.Bucket.fromBucketName(scope, 'InstallerBucket', core_1.Token.asString(deadlineResource.getAtt('S3Bucket'))),
            },
        };
    }
    /**
     * Construct the full version string for the linux patch release referenced in this version object.
     *
     * This is constructed by joining the major, minor,
     * release, and patch versions by dots.
     */
    linuxFullVersionString() {
        const major = core_1.Token.isUnresolved(this.majorVersion) ? core_1.Token.asString(this.majorVersion) : this.majorVersion.toString();
        const minor = core_1.Token.isUnresolved(this.minorVersion) ? core_1.Token.asString(this.minorVersion) : this.minorVersion.toString();
        const release = core_1.Token.isUnresolved(this.releaseVersion) ? core_1.Token.asString(this.releaseVersion) : this.releaseVersion.toString();
        const patch = core_1.Token.isUnresolved(this.linuxInstallers.patchVersion)
            ? core_1.Token.asString(this.linuxInstallers.patchVersion)
            : this.linuxInstallers.patchVersion.toString();
        return `${major}.${minor}.${release}.${patch}`;
    }
    /**
     * Check if we have a full version in the supplied version string. If we don't, we want to make sure the Lambda
     * that fetches the full version number and the installers for it is always run. This allows for Deadline updates
     * to be discovered.
     */
    forceRun(version) {
        return !this.isFullVersion(version) ? crypto_1.randomBytes(32).toString('base64').slice(0, 32) : undefined;
    }
    /**
     * Checks if the supplied version contains the major, minor, release, and patch version numbers,
     * and returns true only if all 4 are supplied.
     */
    isFullVersion(version) {
        const components = version === null || version === void 0 ? void 0 : version.split('.').map(x => parseInt(x));
        if (!components || (components === null || components === void 0 ? void 0 : components.length) != 4) {
            return false;
        }
        return true;
    }
}
exports.VersionQuery = VersionQuery;
//# sourceMappingURL=data:application/json;base64,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