/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { DockerImageAsset } from '@aws-cdk/aws-ecr-assets';
import { Construct } from '@aws-cdk/core';
import { IVersion } from './';
/**
 * Build arguments to supply to a Docker image build.
 */
export interface BuildArgs {
    readonly [name: string]: string;
}
/**
 * Docker container image recipe.
 */
export interface Recipe {
    /**
     * Title of the recipe.
     */
    readonly title: string;
    /**
     * Description of the recipe.
     */
    readonly description: string;
    /**
     * Optional target for a Docker multi-stage build.
     *
     * @default The last stage in the Dockerfile is used
     */
    readonly target?: string;
    /**
     * The Docker build arguments for the recipe.
     *
     * @default No build arguments are supplied
     */
    readonly buildArgs?: BuildArgs;
}
/**
 * A collection of Deadline Docker recipes.
 */
export interface DeadlineDockerRecipes {
    /**
     * A mapping of name to recipe
     */
    readonly [name: string]: Recipe | undefined;
}
/**
 * The manifest included with Deadline Docker image recipes.
 */
export interface Manifest {
    /**
     * The manifest schema version number.
     */
    readonly schema: number;
    /**
     * The version of Deadline that the staging directory contains.
     */
    readonly version: string;
    /**
     * The recipes.
     */
    readonly recipes: DeadlineDockerRecipes;
}
/**
 * Constructor properties of the {@link Stage} class.
 */
export interface StageProps {
    /**
     * The path to the directory where Deadline is staged.
     */
    readonly path: string;
    /**
     * The parsed manifest that describes the contents of the stage directory.
     */
    readonly manifest: Manifest;
}
/**
 * Class for interacting with the Deadline stage directory.
 *
 * The stage is a directory that conforms to a conventional structure that RFDK
 * requires to deploy Deadline. It should contain a manifest file, the Deadline
 * installers, and any supporting files required for building the Deadline
 * container.
 *
 * Note: Current version of RFDK supports Deadline v10.1.9.2 and later.
 */
export declare class Stage {
    /**
     * Loads and parses the manifest file from a given path.
     *
     * @param manifestPath The path to the manifest JSON file.
     */
    static loadManifest(manifestPath: string): Manifest;
    /**
     * Returns a {@link Stage} loaded using the specified directory as the Docker build context and loads and uses the manifest named `manifest.json` in the directory.
     *
     * @param stagePath The path to the Deadline stage directory.
     */
    static fromDirectory(stagePath: string): Stage;
    /**
     * The path to the manifest JSON file relative to the {@link Stage}'s directory'.
     */
    private static readonly MANIFEST_REL_PATH;
    /**
     * Ensures that the manifest is a valid manifest object schema.
     *
     * If the object does not fit the schema, then an Error is thrown.
     *
     * @param rawManifest The parsed manifest object
     */
    private static validateManifest;
    /**
     * The path to the stage directory.
     */
    readonly dirPath: string;
    /**
     * The parsed manifest within the stage directory.
     */
    readonly manifest: Manifest;
    /**
     * Constructs a Stage instance.
     */
    protected constructor(props: StageProps);
    /**
     * Creates a {@link Version} based on the manifest version.
     *
     * @param scope The parent scope.
     * @param id The construct ID.
     */
    getVersion(scope: Construct, id: string): IVersion;
    /**
     * Construct a {@link DockerImageAsset} instance from a recipe in the Stage.
     *
     * @param scope The scope for the {@link DockerImageAsset}.
     * @param id The construct ID of the {@link DockerImageAsset}.
     * @param recipeName The name of the recipe.
     */
    imageFromRecipe(scope: Construct, id: string, recipeName: string): DockerImageAsset;
    /**
     * This removes the patch version from a full version string. No validation is done as that is handled
     * in the constructor with the version check.
     */
    private getReleaseVersion;
}
