"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const path = require("path");
const url_1 = require("url");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const database_connection_1 = require("./database-connection");
/**
 * This construct represents the main Deadline Repository which contains the central database and file system that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository. If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * Resources Deployed
 * ------------------------
 * - Encrypted Amazon Elastic File System (EFS) - If no file system is provided.
 * - An Amazon DocumentDB - If no database connection is provided.
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Repository installation logs.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The file system that is created by, or provided to, this construct contains the data for Deadline's Repository file
 *    system. This file system contains information about your submitted jobs, and the plugin scripts that are run by the
 *    Deadline applications in your render farm. An actor that can modify the contents of this file system can cause your
 *    Deadline applications to run code of their choosing. You should restrict access to this file system to only those who
 *    require it.
 * - The database that is created by, or provided to, this construct is used by Deadline to store data about its configuration,
 *    submitted jobs, machine information and status, and so on. An actor with access to this database can read any information
 *    that is entered into Deadline, and modify the bevavior of your render farm. You should restrict access to this database
 *    to only those who require it.
 */
class Repository extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q;
        super(scope, id);
        if (props.database && ((_a = props.backupOptions) === null || _a === void 0 ? void 0 : _a.databaseRetention)) {
            this.node.addWarning('Backup retention for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem && ((_b = props.removalPolicy) === null || _b === void 0 ? void 0 : _b.filesystem)) {
            this.node.addWarning('RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
        }
        if (props.database && ((_c = props.removalPolicy) === null || _c === void 0 ? void 0 : _c.database)) {
            this.node.addWarning('RemovalPolicy for database will not be applied since a database is not being created by this construct');
        }
        this.version = props.version;
        // Set up the Filesystem and Database components of the repository
        this.fileSystem = (_d = props.fileSystem) !== null && _d !== void 0 ? _d : new core_2.MountableEfs(this, {
            filesystem: new aws_efs_1.FileSystem(this, 'FileSystem', {
                vpc: props.vpc,
                vpcSubnets: (_e = props.vpcSubnets) !== null && _e !== void 0 ? _e : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
                encrypted: true,
                lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_14_DAYS,
                removalPolicy: (_g = (_f = props.removalPolicy) === null || _f === void 0 ? void 0 : _f.filesystem) !== null && _g !== void 0 ? _g : core_1.RemovalPolicy.RETAIN,
            }),
        });
        if (props.database) {
            this.databaseConnection = props.database;
            if (props.databaseAuditLogging !== undefined) {
                const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
                    'Please ensure that the Database provided is configured correctly.';
                this.node.addWarning(warningMsg);
            }
        }
        else {
            const databaseAuditLogging = (_h = props.databaseAuditLogging) !== null && _h !== void 0 ? _h : true;
            /**
             * This option is part of enabling audit logging for DocumentDB; the other required part is the enabling of the CloudWatch exports below.
             *
             * For more information about audit logging in DocumentDB, see:  https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html
             */
            const parameterGroup = databaseAuditLogging ? new aws_docdb_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: 'DocDB cluster parameter group with enabled audit logs',
                family: 'docdb3.6',
                parameters: {
                    audit_logs: 'enabled',
                },
            }) : undefined;
            const instances = (_j = props.documentDbInstanceCount) !== null && _j !== void 0 ? _j : Repository.DEFAULT_NUM_DOCDB_INSTANCES;
            const dbCluster = new aws_docdb_1.DatabaseCluster(this, 'DocumentDatabase', {
                masterUser: { username: 'DocDBUser' },
                instanceProps: {
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.LARGE),
                    vpc: props.vpc,
                    vpcSubnets: (_k = props.vpcSubnets) !== null && _k !== void 0 ? _k : { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true },
                },
                instances,
                backup: {
                    retention: (_m = (_l = props.backupOptions) === null || _l === void 0 ? void 0 : _l.databaseRetention) !== null && _m !== void 0 ? _m : Repository.DEFAULT_DATABASE_RETENTION_PERIOD,
                },
                parameterGroup,
                removalPolicy: (_p = (_o = props.removalPolicy) === null || _o === void 0 ? void 0 : _o.database) !== null && _p !== void 0 ? _p : core_1.RemovalPolicy.RETAIN,
            });
            if (databaseAuditLogging) {
                /**
                 * This option enable export audit logs to Amazon CloudWatch.
                 * This is second options that required for enable audit log.
                 */
                const cfnDB = dbCluster.node.findChild('Resource');
                cfnDB.enableCloudwatchLogsExports = ['audit'];
            }
            /* istanbul ignore next */
            if (!dbCluster.secret) {
                /* istanbul ignore next */
                throw new Error('DBCluster failed to get set up properly -- missing login secret.');
            }
            // This is a workaround because of the bug in CDK implementation:
            // autoMinorVersionUpgrade should be true by default but it's not.
            // This code can be removed once fixed in CDK.
            for (let i = 1; i <= instances; i++) {
                const docdbInstance = dbCluster.node.tryFindChild(`Instance${i}`);
                docdbInstance.autoMinorVersionUpgrade = true;
            }
            this.databaseConnection = database_connection_1.DatabaseConnection.forDocDB({
                database: dbCluster,
                login: dbCluster.secret,
            });
        }
        // Launching the instance which installs the deadline repository in the stack.
        this.installerGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Installer', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            vpcSubnets: (_q = props.vpcSubnets) !== null && _q !== void 0 ? _q : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            minCapacity: 1,
            maxCapacity: 1,
            resourceSignalTimeout: (props.repositoryInstallationTimeout || core_1.Duration.minutes(15)),
            updateType: aws_autoscaling_1.UpdateType.REPLACING_UPDATE,
            replacingUpdateMinSuccessfulInstancesPercent: 100,
        });
        this.node.defaultChild = this.installerGroup;
        // Ensure the DB is serving before we try to connect to it.
        this.databaseConnection.addChildDependency(this.installerGroup);
        // Updating the user data with installation logs stream -- ALWAYS DO THIS FIRST.
        this.configureCloudWatchLogStream(this.installerGroup, `${id}`, props.logGroupProps);
        this.setupDirectConnect(this.installerGroup, Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH);
        this.rootPrefix = props.repositoryInstallationPrefix || Repository.DEFAULT_REPO_PREFIX;
        if (path.posix.isAbsolute(this.rootPrefix)) {
            // If the input path is absolute, then we make it relative (to the root of the repo file-system)
            this.rootPrefix = path.posix.relative(path.posix.sep, this.rootPrefix);
        }
        const repositoryInstallationPath = path.posix.normalize(path.posix.join(Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH, this.rootPrefix));
        // Updating the user data with deadline repository installation commands.
        this.configureRepositoryInstallerScript(this.installerGroup, repositoryInstallationPath, props.version);
        this.configureSelfTermination();
        // Updating the user data with successful cfn-signal commands.
        this.installerGroup.userData.addSignalOnExitCommand(this.installerGroup);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Configures an ECS Container Instance and Task Definition for deploying a Deadline Client that directly connects to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *
     * @inheritdoc true
     */
    configureClientECS(props) {
        var _a, _b;
        const hostMountPoint = (_a = props.containerInstances.filesystemMountPoint) !== null && _a !== void 0 ? _a : '/mnt/repo';
        const containerMountPoint = (_b = props.containers.filesystemMountPoint) !== null && _b !== void 0 ? _b : `/opt/Thinkbox/DeadlineRepository${this.version.majorVersion}`;
        // Set up a direct connection on the host machine. This:
        //  - grants IAM permissions to the role associated with the instance profile access to
        //    - the file-system
        //    - the DB secret containing the credentials
        //  - adds a security group ingress rule to the DB cluster and file-system
        //  - adds userdata commands to mount the repository file-system on the host
        props.containerInstances.hosts.forEach(host => {
            this.setupDirectConnect(host, hostMountPoint);
        });
        // Build up a mapping of environment variables that are used to configure the container's direct connection to the
        // repository
        const containerEnvironment = {
            REPO_URI: url_1.pathToFileURL(containerMountPoint).toString(),
        };
        // The role associated with the task definition needs access to connect to the database
        this.databaseConnection.grantRead(props.containers.taskDefinition.taskRole);
        // Add any environment variables specified by the connection
        Object.entries(this.databaseConnection.containerEnvironment).forEach((entry) => {
            const [envVarName, envVarValue] = entry;
            containerEnvironment[envVarName] = envVarValue;
        });
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.containers.taskDefinition.node.addDependency(this);
        // Configure a named volume in the task-definition that points to the container host's mount-point of the repository
        // file-system
        props.containers.taskDefinition.addVolume({
            name: Repository.ECS_VOLUME_NAME,
            host: {
                sourcePath: path.posix.normalize(path.posix.join(hostMountPoint, this.rootPrefix)),
            },
        });
        // Return the container connection. This data structure contains all the pieces needed to create containers
        // that can directly connect to the repository.
        return {
            containerEnvironment,
            readOnlyMountPoint: {
                containerPath: containerMountPoint,
                readOnly: true,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
            readWriteMountPoint: {
                containerPath: containerMountPoint,
                readOnly: false,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
        };
    }
    /**
     * Configure a Deadline Client, that is running in an Amazon EC2 instance, for direct connection to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *    - Configures Deadline to direct-connect to the Repository.
     *
     * @inheritdoc true
     */
    configureClientInstance(props) {
        var _a;
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.host.node.addDependency(this);
        this.setupDirectConnect(props.host, props.mountPoint);
        const stack = core_1.Stack.of(this);
        const uuid = 'f625e47b-7aed-4879-9861-513a72145525';
        const uniqueId = 'DeadlineRepository' + props.host.osType + uuid.replace(/[-]/g, '');
        const configureDirectConnect = (_a = stack.node.tryFindChild(uniqueId)) !== null && _a !== void 0 ? _a : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: props.host.osType,
            baseName: 'configureRepositoryDirectConnect',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        configureDirectConnect.grantRead(props.host);
        this.databaseConnection.addConnectionDBArgs(props.host);
        const repoPath = path.posix.normalize(path.posix.join(props.mountPoint, this.rootPrefix));
        configureDirectConnect.executeOn({
            host: props.host,
            args: [`"${repoPath}"`],
        });
    }
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    setupDirectConnect(host, repositoryMountPoint) {
        if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            throw new Error('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
        }
        this.databaseConnection.grantRead(host);
        this.databaseConnection.allowConnectionsFrom(host);
        this.fileSystem.mountToLinuxInstance(host, {
            location: repositoryMountPoint,
        });
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    configureCloudWatchLogStream(installerGroup, groupName, logGroupProps) {
        var _a;
        const prefix = (_a = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _a !== void 0 ? _a : Repository.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'RepositoryLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(installerGroup);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'deadlineRepositoryInstallationLogs', '/tmp/bitrock_installer.log');
        new core_2.CloudWatchAgent(this, 'RepositoryInstallerLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: installerGroup,
        });
    }
    configureSelfTermination() {
        const tagKey = 'resourceLogicalId';
        /*
        Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources
        as it will cause cyclic dependency. Hence, using Condition Keys
        */
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${tagKey}`] = this.node.uniqueId;
        core_1.Tags.of(this.installerGroup).add(tagKey, this.node.uniqueId);
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.installerGroup.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and asg name and then setting all the capacity to 0 to terminate the installer.
        this.installerGroup.userData.addOnExitCommands('INSTANCE="$(curl http://169.254.169.254/latest/meta-data/instance-id)"');
        this.installerGroup.userData.addOnExitCommands('ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"');
        this.installerGroup.userData.addOnExitCommands('aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
    configureRepositoryInstallerScript(installerGroup, installPath, version) {
        const installerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'DeadlineRepositoryInstallerScript', {
            osType: installerGroup.osType,
            baseName: 'installDeadlineRepository',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        this.databaseConnection.addInstallerDBArgs(installerGroup);
        version.linuxInstallers.repository.s3Bucket.grantRead(installerGroup, version.linuxInstallers.repository.objectKey);
        installerScriptAsset.executeOn({
            host: installerGroup,
            args: [
                `"s3://${version.linuxInstallers.repository.s3Bucket.bucketName}/${version.linuxInstallers.repository.objectKey}"`,
                `"${installPath}"`,
                version.linuxFullVersionString(),
            ],
        });
    }
}
exports.Repository = Repository;
/**
 * Default file system mount path for repository
 */
Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH = '/mnt/efs/fs1';
/**
 * Default installation prefix for deadline repository.
 */
Repository.DEFAULT_REPO_PREFIX = 'DeadlineRepository';
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
Repository.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * How often Cloudwatch logs will be flushed.
 */
Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
 * containers.
 */
Repository.ECS_VOLUME_NAME = 'RepositoryFilesystem';
/**
 * The default number of DocDB instances if one isn't provided in the props.
 */
Repository.DEFAULT_NUM_DOCDB_INSTANCES = 1;
/**
 * Default retention period for DocumentDB automated backups if one isn't provided in the props.
 */
Repository.DEFAULT_DATABASE_RETENTION_PERIOD = core_1.Duration.days(15);
//# sourceMappingURL=data:application/json;base64,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