"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseConnection = void 0;
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
/**
 * Helper class for connecting Thinkbox's Deadline to a specific Database.
 */
class DatabaseConnection {
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to Amazon DocumentDB.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     */
    static forDocDB(options) {
        return new DocDBDatabaseConnection(options);
    }
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to MongoDB.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     */
    static forMongoDbInstance(options) {
        return new MongoDbInstanceDatabaseConnection(options);
    }
}
exports.DatabaseConnection = DatabaseConnection;
/**
 * Specialization of {@link DatabaseConnection} targetting Amazon DocumentDB.
 */
class DocDBDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        this.containerEnvironment = {
            // The container must fetch the credentials from Secrets Manager
            DB_CREDENTIALS_URI: this.props.login.secretArn,
        };
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${core_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', `INSTALLER_DB_ARGS=( ["--dbuser"]=$DB_USERNAME ["--dbpassword"]=$DB_PASSWORD ["--dbhost"]=${this.props.database.clusterEndpoint.hostname}` +
            ` ["--dbport"]=${this.props.database.clusterEndpoint.portAsString()} ["--dbtype"]=DocumentDB )`, 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${core_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', 'sudo -u ec2-user "${deadlinecommand}" -StoreDatabasecredentials "${DB_USERNAME}" "${DB_PASSWORD}"', 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, this.props.database.connections.defaultPort);
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.login.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        // To depend on document DB it is not sufficient to depend on the Cluster. The instances are what serves data, so
        // we must add a dependency to an instance in the DocDB cluster.
        // The DocDB L2 does not expose any of its instances as properties, so we have to escape-hatch to gain access.
        const docdbInstance = this.props.database.node.tryFindChild('Instance1');
        // We won't find an instance in two situations:
        //  1) The DocDB Cluster was created from attributes. In this case, the DocDB pre-exists the stack and there's no need
        //     to depend on anything.
        //  2) The DocDB Cluster was constructed, but the internal name for the instance has been changed from 'Instance1'; this is
        //     unlikely, but not impossible.
        // We can differentiate cases (1) & (2) by looking for the defaultChild on the cluster. The version from attributes will not have one.
        if (docdbInstance) {
            child.node.addDependency(docdbInstance);
        }
        else if (this.props.database.node.defaultChild) {
            throw new Error('The internal implementation of the AWS CDK\'s DocumentDB cluster construct may have changed. Please update to a newer RFDK for an updated implementation, or file a ticket if this is the latest release.');
        }
    }
}
/**
 * Specialization of {@link DatabaseConnection} targetting MongoDB.
 */
class MongoDbInstanceDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        this.containerEnvironment = {
            DB_TLS_CLIENT_CERT_URI: props.clientCertificate.cert.secretArn,
            DB_TLS_CLIENT_CERT_PASSWORD_URI: props.clientCertificate.passphrase.secretArn,
        };
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, aws_ec2_1.Port.tcp(this.props.database.port));
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 
        // Suppress -x, so no secrets go to the logs
        'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${core_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'INSTALLER_DB_ARGS=( ["--dbssl"]=true ["--dbauth"]=true ["--dbsslauth"]=true ' +
            `["--dbhost"]="${this.props.database.fullHostname}" ["--dbport"]=${this.props.database.port} ` +
            `["--dbclientcert"]="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}" ["--dbcertpass"]=$CERT_PASSWORD )`, 'unset CERT_PASSWORD', 
        // Restore -x, if it was set.
        'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export DB_CERT_FILE="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}"`, `export DB_CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${core_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.clientCertificate.cert.grantRead(grantee);
        this.props.clientCertificate.passphrase.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        if (this.props.database.hasOwnProperty('server')) {
            const db = this.props.database;
            child.node.addDependency(db.server.autoscalingGroup.node.defaultChild);
        }
    }
    /**
     * Download the client PKCS#12 certificate for authenticating to the MongoDB, and place it into
     * the path defined by: DB_CERT_LOCATION
     * @param host
     */
    downloadCertificate(host) {
        var _a;
        const stack = core_1.Stack.of(host);
        const uuid = 'e8125dd2-ab2c-4861-8ee4-998c26b30ee0';
        const uniqueId = 'GetSecretToFile' + host.osType + uuid.replace(/[-]/g, '');
        const getSecretsScript = (_a = stack.node.tryFindChild(uniqueId)) !== null && _a !== void 0 ? _a : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: host.osType,
            baseName: 'getSecretToFile',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        getSecretsScript.executeOn({
            host,
            args: [
                this.props.clientCertificate.cert.secretArn,
                MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION,
            ],
        });
    }
}
MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION = '/opt/Thinkbox/certs/mongo_client.pfx';
//# sourceMappingURL=data:application/json;base64,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