"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScriptAsset = void 0;
const path = require("path");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
var ScriptExtension;
(function (ScriptExtension) {
    ScriptExtension[ScriptExtension[".sh"] = 0] = ".sh";
    ScriptExtension[ScriptExtension[".ps1"] = 1] = ".ps1";
})(ScriptExtension || (ScriptExtension = {}));
var ScriptPathPrefix;
(function (ScriptPathPrefix) {
    ScriptPathPrefix[ScriptPathPrefix["bash"] = 0] = "bash";
    ScriptPathPrefix[ScriptPathPrefix["powershell"] = 1] = "powershell";
})(ScriptPathPrefix || (ScriptPathPrefix = {}));
/**
 * This method returns the path to a script based on RFDK conventional directory structure and the target
 * operating system of the script.
 *
 * The directory structure convention keeps linux scripts in `${scriptRoot}//bash/*.sh` and Windows scripts in
 * `${scriptRoot}/powershell/*.ps1`.
 *
 * @param osType
 * @param scriptName
 */
function getConventionalScriptPath(params) {
    const { rootDir: scriptDir, baseName: scriptName, osType } = params;
    // Make sure we have a known osType. The error message is pretty obtuse if we don't:
    //  The "path" argument must be of type string. Received undefined
    if (ScriptPathPrefix[osType] === undefined || ScriptExtension[osType] == undefined) {
        throw Error(`Unknown osType: ${osType}`);
    }
    return path.join(scriptDir, ScriptPathPrefix[osType], `${scriptName}${ScriptExtension[osType]}`);
}
/**
 * An S3 asset that contains a shell script intended to be executed through instance user data.
 *
 * This is used by other constructs to generalize the concept of a script
 * (bash or powershell) that executes on an instance.
 * It provides a wrapper around the CDK’s S3 Asset construct
 * ( https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-s3-assets.Asset.html )
 *
 * The script asset is placed into and fetched from the CDK bootstrap S3 bucket.
 *
 * Resources Deployed
 * ------------------------
 * - An Asset which is uploaded to the bootstrap S3 bucket.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 */
class ScriptAsset extends aws_s3_assets_1.Asset {
    /**
     * Returns a {@link ScriptAsset} instance by computing the path to the script using RFDK's script directory structure convention.
     *
     * By convention, scripts are kept in a `scripts` directory in each `aws-rfdk/*` package. The scripts are organized
     * based on target shell (and implicitly target operating system). The directory structure looks like:
     *
     * ```
     * scripts/
     *    bash/
     *      script-one.sh
     *      script-two.sh
     *    powershell
     *      script-one.ps1
     *      script-one.ps1
     * ```
     *
     * @param scope The scope for the created {@link ScriptAsset}.
     * @param id The construct id for the created {@link ScriptAsset}.
     * @param scriptParams The parameters that are used to compute the conventional path to the script file.
     */
    static fromPathConvention(scope, id, scriptParams) {
        const scriptPath = getConventionalScriptPath(scriptParams);
        return new ScriptAsset(scope, id, { path: scriptPath });
    }
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * Adds commands to the {@link IScriptHost} to download and execute the ScriptAsset.
     *
     * @param props The parameters for executing the script.
     */
    executeOn(props) {
        const { host, args } = props;
        // Grant permission to fetch the script asset
        this.grantRead(host);
        // Add a command to the user data to download the script asset
        const instanceScriptPath = host.userData.addS3DownloadCommand({
            bucket: this.bucket,
            bucketKey: this.s3ObjectKey,
        });
        // Add a command to the user data to execute the downloaded script
        host.userData.addExecuteFileCommand({
            filePath: instanceScriptPath,
            arguments: (args || []).join(' '),
        });
    }
}
exports.ScriptAsset = ScriptAsset;
//# sourceMappingURL=data:application/json;base64,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