"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImportedAcmCertificate = void 0;
const crypto = require("crypto");
const path_1 = require("path");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../lambdas/lambdaLayerVersionArns");
/**
 * A Construct that creates an AWS CloudFormation Custom Resource that models a certificate that is imported into AWS Certificate Manager (ACM).
 *
 * It uses an AWS Lambda Function to extract the certificate from Secrets in AWS SecretsManager
 * and then import it into ACM. The interface is intended to be used with the {@link X509CertificatePem} Construct.
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * - An AWS Lambda Function, with IAM Role - Used to create/update/delete the Custom Resource.
 * - AWS Certificate Manager Certificate - Created by the Custom Resource.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda for this construct also has broad IAM permissions to delete any Certificate that is stored
 *    in AWS Certificate Manager. You should not grant any additional actors/principals the ability to modify or
 *    execute this Lambda.
 */
class ImportedAcmCertificate extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.stack = core_1.Stack.of(this);
        this.env = {
            account: this.stack.account,
            region: this.stack.region,
        };
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            serverSideEncryption: true,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lambda = new aws_lambda_1.SingletonFunction(this, 'AcmImporter', {
            uuid: ImportedAcmCertificate.IMPORTER_UUID,
            code: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', 'lambdas', 'nodejs')),
            handler: 'x509-certificate.importCert',
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            layers: [openSslLayer],
            retryAttempts: 0,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            timeout: core_1.Duration.seconds(30),
        });
        this.database.grantReadWriteData(lambda);
        this.database.grant(lambda, 'dynamodb:DescribeTable');
        props.cert.grantRead(lambda);
        props.key.grantRead(lambda);
        props.passphrase.grantRead(lambda);
        (_a = props.certChain) === null || _a === void 0 ? void 0 : _a.grantRead(lambda);
        (_b = props.encryptionKey) === null || _b === void 0 ? void 0 : _b.grantDecrypt(lambda);
        const uniqueValue = crypto.createHash('md5').update(this.node.uniqueId).digest('hex');
        this.uniqueTag = new core_1.Tag(`AcmCertImport-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`aws:RequestTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'acm:AddTagsToCertificate',
                'acm:ImportCertificate',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // GetCertificate and DeleteCertificate don't currently support and conditions, so we have to give a broader policy
        // on them for now.
        // See https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanager.html#awscertificatemanager-aws_TagKeys
        // for the condition keys currently available on ACM actions.
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'acm:DeleteCertificate',
                'acm:DescribeCertificate',
                'acm:GetCertificate',
            ],
            resources: ['*'],
        }));
        const properties = {
            X509CertificatePem: {
                Cert: props.cert.secretArn,
                Key: props.key.secretArn,
                Passphrase: props.passphrase.secretArn,
                CertChain: props.certChain ? props.certChain.secretArn : '',
            },
            Tags: [
                {
                    Key: this.uniqueTag.key,
                    Value: this.uniqueTag.value,
                },
                { Key: 'Name',
                    Value: this.uniqueTag.value,
                },
            ],
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: lambda.functionArn,
            properties,
            resourceType: 'Custom::RFDK_AcmImportedCertificate',
        });
        this.certificateArn = core_1.Token.asString(resource.getAtt('CertificateArn'));
    }
}
exports.ImportedAcmCertificate = ImportedAcmCertificate;
ImportedAcmCertificate.IMPORTER_UUID = '2d20d8f2-7b84-444e-b738-c75b499a9eaa';
//# sourceMappingURL=data:application/json;base64,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