"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectableApplicationEndpoint = exports.ApplicationEndpoint = exports.Endpoint = void 0;
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const core_1 = require("@aws-cdk/core");
/**
 * Connection endpoint.
 *
 * Consists of a combination of hostname, port, and transport protocol.
 */
class Endpoint {
    /**
     * Constructs an Endpoint instance.
     *
     * @param props The properties for the endpoint.
     */
    constructor(props) {
        var _a;
        const { address, port } = props;
        const protocol = (_a = props.protocol) !== null && _a !== void 0 ? _a : aws_ec2_1.Protocol.TCP;
        if (!core_1.Token.isUnresolved(port) && !Endpoint.isValidPort(port)) {
            throw new Error(`Port must be an integer between [${Endpoint.MIN_PORT}, ${Endpoint.MAX_PORT}] but got: ${port}`);
        }
        this.hostname = address;
        this.portNumber = port;
        this.protocol = protocol;
        this.port = new aws_ec2_1.Port({
            protocol: this.protocol,
            fromPort: port,
            toPort: port,
            stringRepresentation: this.renderPort(port),
        });
        this.socketAddress = `${address}:${this.portAsString()}`;
    }
    /**
     * Determines if a port is valid
     *
     * @param port: The port number
     * @returns boolean whether the port is valid
     */
    static isValidPort(port) {
        return Number.isInteger(port) && port >= Endpoint.MIN_PORT && port <= Endpoint.MAX_PORT;
    }
    /**
     * Returns the port number as a string representation that can be used for embedding within other strings.
     *
     * This is intended to deal with CDK's token system. Numeric CDK tokens are not expanded when their string
     * representation is embedded in a string. This function returns the port either as an unresolved string token or
     * as a resolved string representation of the port value.
     *
     * @returns An (un)resolved string representation of the endpoint's port number
     */
    portAsString() {
        if (core_1.Token.isUnresolved(this.portNumber)) {
            return core_1.Token.asString(this.portNumber);
        }
        else {
            return this.portNumber.toString();
        }
    }
    renderPort(port) {
        return core_1.Token.isUnresolved(port) ? '{IndirectPort}' : port.toString();
    }
}
exports.Endpoint = Endpoint;
/**
 * The minimum port value
 */
Endpoint.MIN_PORT = 1;
/**
 * The maximum port value
 */
Endpoint.MAX_PORT = 65535;
/**
 * An endpoint serving http or https for an application.
 */
class ApplicationEndpoint extends Endpoint {
    /**
     * Constructs a {@link ApplicationEndpoint} instance.
     *
     * @param props The properties for the application endpoint.
     */
    constructor(props) {
        var _a;
        super({
            address: props.address,
            port: props.port,
            protocol: aws_ec2_1.Protocol.TCP,
        });
        this.applicationProtocol = (_a = props.protocol) !== null && _a !== void 0 ? _a : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
    }
}
exports.ApplicationEndpoint = ApplicationEndpoint;
/**
 * An endpoint serving http or https for an application.
 */
class ConnectableApplicationEndpoint extends ApplicationEndpoint {
    /**
     * Constructs a {@link ApplicationEndpoint} instance.
     *
     * @param props The properties for the application endpoint.
     */
    constructor(props) {
        super(props);
        this.connections = props.connections;
    }
}
exports.ConnectableApplicationEndpoint = ConnectableApplicationEndpoint;
//# sourceMappingURL=data:application/json;base64,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