"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudWatchConfigBuilder = exports.TimeZone = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * Enum to describe the time zone property.
 */
var TimeZone;
(function (TimeZone) {
    TimeZone["LOCAL"] = "Local";
    TimeZone["UTC"] = "UTC";
})(TimeZone = exports.TimeZone || (exports.TimeZone = {}));
/**
 * Class that can build a CloudWatch Agent configuration.
 */
class CloudWatchConfigBuilder {
    /**
     * Constructs.
     */
    constructor(logFlushInterval = CloudWatchConfigBuilder.DEFAULT_FLUSH_INTERVAL) {
        this.cloudWatchFileList = [];
        this.logFlushInterval = logFlushInterval;
    }
    /**
     * This method adds the log file path and its associated log group and log stream properties to the list of files which needs to be streamed to cloud watch logs.
     *
     * @param logGroupName - string for the log group name.
     * @param logStreamPrefix - string for the log stream prefix.
     * @param logFilePath - local file path which needs to be streamed.
     * @param timeZone -  the time zone to use when putting timestamps on log events.
     */
    addLogsCollectList(logGroupName, logStreamPrefix, logFilePath, timeZone = CloudWatchConfigBuilder.DEFAULT_LOG_TIMEZONE) {
        this.cloudWatchFileList.push({
            log_group_name: logGroupName,
            log_stream_name: logStreamPrefix + '-' + CloudWatchConfigBuilder.CLOUDWATCH_CONFIG_INSTANCE_ID_VARIABLE,
            file_path: logFilePath,
            timezone: timeZone,
        });
    }
    /**
     * The method generates the configuration for log file streaming to be added to CloudWatch Agent Configuration File.
     */
    generateCloudWatchConfiguration() {
        const cloudWatchConfig = {
            logs: {
                logs_collected: {
                    files: {
                        collect_list: this.cloudWatchFileList,
                    },
                },
                log_stream_name: CloudWatchConfigBuilder.DEFAULT_STREAM_NAME
                    + '-'
                    + CloudWatchConfigBuilder.CLOUDWATCH_CONFIG_INSTANCE_ID_VARIABLE,
                force_flush_interval: this.logFlushInterval.toSeconds(),
            },
        };
        return JSON.stringify(cloudWatchConfig);
    }
}
exports.CloudWatchConfigBuilder = CloudWatchConfigBuilder;
CloudWatchConfigBuilder.DEFAULT_STREAM_NAME = 'DefaultLogStream';
CloudWatchConfigBuilder.DEFAULT_FLUSH_INTERVAL = core_1.Duration.seconds(60);
CloudWatchConfigBuilder.DEFAULT_LOG_TIMEZONE = TimeZone.LOCAL;
CloudWatchConfigBuilder.CLOUDWATCH_CONFIG_INSTANCE_ID_VARIABLE = '{instance_id}';
//# sourceMappingURL=data:application/json;base64,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