"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudWatchAgent = void 0;
const path = require("path");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_ssm_1 = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const script_assets_1 = require("./script-assets");
/**
 * This construct is a thin wrapper that provides the ability to install and configure the CloudWatchAgent ( https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html ) on one or more EC2 instances during instance startup.
 *
 * It accomplishes this by downloading and executing the configuration script on the instance.
 * The script will download the CloudWatch Agent installer,
 * optionally verify the installer, and finally install the CloudWatch Agent.
 * The installer is downloaded via the Amazon S3 API, thus, this construct can be used
 * on instances that have no access to the internet as long as the VPC contains
 * an VPC Gateway Endpoint for S3 ( https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-s3.html ).
 *
 * {@link CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR} - Context variable to skip validation
 * of the downloaded CloudWatch Agent installer if set to 'TRUE'.
 * WARNING: Only use this if your deployments are failing due to a validation failure,
 * but you have verified that the failure is benign.
 *
 * Resources Deployed
 * ------------------------
 * - String SSM Parameter in Systems Manager Parameter Store to store the cloudwatch agent configuration;
 * - A script Asset which is uploaded to S3 bucket.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 */
class CloudWatchAgent extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Create the asset for the configuration script
        this.configurationScript = script_assets_1.ScriptAsset.fromPathConvention(scope, 'CloudWatchConfigurationScriptAsset', {
            osType: props.host.osType,
            baseName: 'configureCloudWatchAgent',
            rootDir: path.join(__dirname, '../scripts/'),
        });
        // Create a new SSM Parameter holding the json configuration
        this.ssmParameterForConfig = new aws_ssm_1.StringParameter(scope, 'StringParameter', {
            description: 'config file for Repository logs config',
            stringValue: props.cloudWatchConfig,
        });
        this.grantRead(props.host);
        this.configure(props.host, this.node.tryGetContext(CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR) === 'TRUE');
    }
    /**
     * Grants read permissions to the principal on the assets bucket and parameter store.
     */
    grantRead(grantee) {
        this.configurationScript.grantRead(grantee);
        this.ssmParameterForConfig.grantRead(grantee);
    }
    /**
     * Configures the CloudWatch Agent on the target host.
     *
     * This is done by adding UserData commands to the target host.
     *
     * @param host The host to configure the CloudWatch agent on
     * @param skipValidation Skips the validation of the CloudWatch agent installer if set to true.
     */
    configure(host, skipValidation) {
        // Grant access to the required CloudWatch Agent installer files
        const cloudWatchAgentBucket = aws_s3_1.Bucket.fromBucketArn(this, 'CloudWatchAgentBucket', 'arn:aws:s3:::amazoncloudwatch-agent');
        cloudWatchAgentBucket.grantRead(host);
        const scriptArgs = [];
        if (skipValidation) {
            // Flags must be set before positional arguments for some scripts
            scriptArgs.push(CloudWatchAgent.SKIP_CWAGENT_VALIDATION_FLAG);
        }
        scriptArgs.push(this.ssmParameterForConfig.parameterName);
        this.configurationScript.executeOn({
            host,
            args: scriptArgs,
        });
    }
}
exports.CloudWatchAgent = CloudWatchAgent;
/**
 * The context variable to indicate that CloudWatch agent installer validation should be skipped.
 */
CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR = 'RFDK_SKIP_CWAGENT_VALIDATION';
/**
 * The flag for configureCloudWatchAgent script to skip CloudWatch agent installer validation.
 */
CloudWatchAgent.SKIP_CWAGENT_VALIDATION_FLAG = '-s';
//# sourceMappingURL=data:application/json;base64,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