"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleCustomResource = void 0;
const hash_1 = require("./hash");
const reply_1 = require("./reply");
/* istanbul ignore file */
class SimpleCustomResource {
    constructor() {
        var _a;
        // Optionally suppress debugging statements.
        this.debugMode = ((_a = process.env.DEBUG) !== null && _a !== void 0 ? _a : 'false') === 'true';
        if (!this.debugMode) {
            console.debug = () => { };
        }
    }
    /**
     * Handler/engine for the CustomResource state machine. Users of this class should
     * instantiate the class, and then immediately call this function.
     * @param event The event passed to the lambda handler.
     * @param context The lambda context passed to the lambda handler.
     */
    async handler(event, context) {
        var _a, _b;
        let status = reply_1.CfnResponseStatus.SUCCESS;
        let failReason;
        let cfnData;
        console.log(`Handling event: ${JSON.stringify(event)}`);
        const requestType = event.RequestType;
        const resourceProperties = (_a = event.ResourceProperties) !== null && _a !== void 0 ? _a : {};
        const physicalId = hash_1.calculateSha256Hash(resourceProperties);
        try {
            if (requestType === 'Create') {
                if (!this.validateInput(resourceProperties)) {
                    throw Error(`Input did not pass validation check. Check log group "${context.logGroupName}" ` +
                        `for log stream ${context.logStreamName} for additional information.`);
                }
                cfnData = await this.doCreate(physicalId, resourceProperties);
                console.debug(`Create data: ${JSON.stringify(cfnData)}`);
            }
            else if (requestType === 'Update') {
                if (!this.validateInput(resourceProperties)) {
                    throw Error('Input did not pass validation check');
                }
                const oldResourceProperties = (_b = event.OldResourceProperties) !== null && _b !== void 0 ? _b : {};
                const oldPhysicalId = hash_1.calculateSha256Hash(oldResourceProperties);
                if (oldPhysicalId !== physicalId) {
                    console.log('Doing Create -- ResoureceProperties differ.');
                    cfnData = await this.doCreate(physicalId, resourceProperties);
                    console.debug(`Update data: ${JSON.stringify(cfnData)}`);
                }
            }
            else {
                await this.doDelete(physicalId, resourceProperties);
            }
        }
        catch (e) {
            // We want to always catch the exception for a CfnCustomResource CloudFormation
            // must be notified about the success/failure of the lambda at all times;
            // failure to notify results in a stuck stack that takes at least an hour to
            // timeout.
            status = reply_1.CfnResponseStatus.FAILED;
            failReason = `${e.message}\n${e.stack}`;
        }
        finally {
            // Always send a response to CloudFormation, signal success or
            // failure based on whether or not we had an exception.
            await reply_1.sendCfnResponse({
                event,
                context,
                status,
                reason: failReason,
                physicalId,
                data: cfnData,
            });
        }
        const response = `${status}` + (failReason !== null && failReason !== void 0 ? failReason : '');
        console.log(`Result: ${response}`);
        return response;
    }
}
exports.SimpleCustomResource = SimpleCustomResource;
//# sourceMappingURL=data:application/json;base64,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