"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RedshiftConnectorProfile = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const type_1 = require("./type");
const appflow_permissions_manager_1 = require("../core/appflow-permissions-manager");
const connector_profile_1 = require("../core/connectors/connector-profile");
class RedshiftConnectorProfile extends connector_profile_1.ConnectorProfileBase {
    static fromConnectionProfileArn(scope, id, arn) {
        return this._fromConnectorProfileAttributes(scope, id, { arn });
    }
    static fromConnectionProfileName(scope, id, name) {
        return this._fromConnectorProfileAttributes(scope, id, { name });
    }
    constructor(scope, id, props) {
        super(scope, id, props, type_1.RedshiftConnectorType.instance);
        this.tryAddNodeDependency(this, props.cluster);
        this._location = props.intermediateLocation;
    }
    buildConnectorProfileProperties(props) {
        const properties = props;
        const redshiftAccessRole = properties.bucketAccessRole ?? this.buildRedshiftAccessRole(this.node.id, properties.cluster, properties.intermediateLocation);
        const appflowDataApiRole = properties.dataApiRole ?? this.buildAppFlowDataApiRole(this.node.id, properties.cluster, properties.databaseName, properties.basicAuth.username);
        this.tryAddNodeDependency(this, redshiftAccessRole);
        this.tryAddNodeDependency(this, appflowDataApiRole);
        this.tryAddNodeDependency(this, properties.intermediateLocation.bucket);
        appflow_permissions_manager_1.AppFlowPermissionsManager.instance().grantBucketReadWrite(properties.intermediateLocation.bucket);
        return {
            redshift: {
                bucketName: properties.intermediateLocation.bucket.bucketName,
                bucketPrefix: properties.intermediateLocation.prefix,
                roleArn: redshiftAccessRole.roleArn,
                clusterIdentifier: properties.cluster.clusterName,
                databaseName: properties.databaseName,
                dataApiRoleArn: appflowDataApiRole.roleArn,
            },
        };
    }
    buildConnectorProfileCredentials(props) {
        const properties = props;
        return {
            redshift: properties && {
                username: properties.basicAuth.username,
                password: properties.basicAuth.password?.unsafeUnwrap(),
            },
        };
    }
    buildRedshiftAccessRole(id, cluster, location) {
        // see: https://docs.aws.amazon.com/appflow/latest/userguide/security_iam_service-role-policies.html#redshift-access-s3
        const role = new aws_iam_1.Role(this.stack, `${id}RedshiftRole`, {
            assumedBy: new aws_iam_1.ServicePrincipal('redshift.amazonaws.com'),
        });
        location.bucket.grantRead(role, location.prefix ? `${location.prefix}/*` : '*');
        const modifierId = `${id}RedshiftRoleAttach`;
        const modifier = new custom_resources_1.AwsCustomResource(this.stack, modifierId, {
            onCreate: {
                service: 'Redshift',
                action: 'modifyClusterIamRoles',
                parameters: {
                    ClusterIdentifier: cluster.clusterName,
                    AddIamRoles: [role.roleArn],
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(modifierId),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromStatements([
                new aws_iam_1.PolicyStatement({
                    actions: ['iam:PassRole'],
                    resources: [role.roleArn],
                    effect: aws_iam_1.Effect.ALLOW,
                }),
                new aws_iam_1.PolicyStatement({
                    actions: ['redshift:ModifyClusterIamRoles'],
                    resources: [
                        `arn:${aws_cdk_lib_1.Aws.PARTITION}:redshift:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:cluster:${cluster.clusterName}`,
                    ],
                    effect: aws_iam_1.Effect.ALLOW,
                }),
            ]),
        });
        this.node.addDependency(modifier);
        modifier.node.addDependency(cluster);
        return role;
    }
    buildAppFlowDataApiRole(id, cluster, databaseName, username) {
        // see: https://docs.aws.amazon.com/appflow/latest/userguide/security_iam_service-role-policies.html#access-redshift
        const role = new aws_iam_1.Role(this.stack, `${id}AppFlowDataApiRole`, {
            assumedBy: new aws_iam_1.ServicePrincipal('appflow.amazonaws.com'),
        });
        const policy = new aws_iam_1.Policy(this.stack, `${id}AppFlowDataApiRolePolicy`, {
            roles: [role],
            statements: [
                new aws_iam_1.PolicyStatement({
                    sid: 'DataAPIPermissions',
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: [
                        'redshift-data:ExecuteStatement',
                        'redshift-data:GetStatementResult',
                        'redshift-data:DescribeStatement',
                    ],
                    resources: ['*'],
                }),
                new aws_iam_1.PolicyStatement({
                    sid: 'GetCredentialsForAPIUser',
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ['redshift:GetClusterCredentials'],
                    resources: [
                        `arn:aws:redshift:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:dbname:${cluster.clusterName}/${databaseName}`,
                        `arn:aws:redshift:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:dbuser:${cluster.clusterName}/${username ?? '*'}`,
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    sid: 'DenyCreateAPIUser',
                    effect: aws_iam_1.Effect.DENY,
                    actions: ['redshift:CreateClusterUser'],
                    resources: [
                        `arn:aws:redshift:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:dbuser:${cluster.clusterName}/*`,
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    sid: 'ServiceLinkedRole',
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ['iam:CreateServiceLinkedRole'],
                    resources: [
                        `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/aws-service-role/redshift-data.amazonaws.com/AWSServiceRoleForRedshift`,
                    ],
                    conditions: {
                        StringLike: {
                            'iam:AWSServiceName': 'redshift-data.amazonaws.com',
                        },
                    },
                }),
            ],
        });
        this.tryAddNodeDependency(this, policy);
        return role;
    }
}
exports.RedshiftConnectorProfile = RedshiftConnectorProfile;
_a = JSII_RTTI_SYMBOL_1;
RedshiftConnectorProfile[_a] = { fqn: "@cdklabs/cdk-appflow.RedshiftConnectorProfile", version: "0.0.32" };
//# sourceMappingURL=data:application/json;base64,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