"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protocol = exports.Service = exports.ServiceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an external IP address, that's outside of your cluster.
 *
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 *
 * @stability stable
 */
var ServiceType;
(function (ServiceType) {
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    ServiceType["NODE_PORT"] = "NodePort";
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 *
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 *
 * @stability stable
 */
class Service extends base_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c, _d;
        super(scope, id);
        this.apiObject = new k8s.KubeService(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this._toKube() }),
        });
        this.clusterIP = props.clusterIP;
        this.externalName = props.externalName;
        if (props.externalName !== undefined) {
            this.type = ServiceType.EXTERNAL_NAME;
        }
        else {
            this.type = (_b = props.type) !== null && _b !== void 0 ? _b : ServiceType.CLUSTER_IP;
        }
        this._externalIPs = (_c = props.externalIPs) !== null && _c !== void 0 ? _c : [];
        this._ports = [];
        this._selector = {};
        this._loadBalancerSourceRanges = props.loadBalancerSourceRanges;
        for (const portAndOptions of (_d = props.ports) !== null && _d !== void 0 ? _d : []) {
            this.serve(portAndOptions.port, portAndOptions);
        }
    }
    /**
     * Returns the labels which are used to select pods for this service.
     *
     * @stability stable
     */
    get selector() {
        return this._selector;
    }
    /**
     * Ports for this service.
     *
     * Use `serve()` to expose additional service ports.
     *
     * @stability stable
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Associate a deployment to this service.
     *
     * If not targetPort is specific in the portOptions, then requests will be routed
     * to the port exposed by the first container in the deployment's pods.
     * The deployment's `labelSelector` will be used to select pods.
     *
     * @param deployment The deployment to expose.
     * @param options Optional settings for the port.
     * @stability stable
     */
    addDeployment(deployment, options = {}) {
        var _b, _c;
        const containers = deployment.containers;
        if (containers.length === 0) {
            throw new Error('Cannot expose a deployment without containers');
        }
        // just a PoC, we assume the first container is the main one.
        // TODO: figure out what the correct thing to do here.
        const container = containers[0];
        const port = (_b = options.port) !== null && _b !== void 0 ? _b : container.port;
        const targetPort = (_c = options.targetPort) !== null && _c !== void 0 ? _c : containers[0].port;
        if (!port) {
            throw new Error('Cannot determine port. Either pass `port` in options or configure a port on the first container of the deployment');
        }
        const selector = Object.entries(deployment.labelSelector);
        if (selector.length === 0) {
            throw new Error('deployment does not have a label selector');
        }
        if (Object.keys(this.selector).length > 0) {
            throw new Error('a selector is already defined for this service. cannot add a deployment');
        }
        for (const [k, v] of selector) {
            this.addSelector(k, v);
        }
        this.serve(port, { ...options, targetPort });
    }
    /**
     * Services defined using this spec will select pods according the provided label.
     *
     * @param label The label key.
     * @param value The label value.
     * @stability stable
     */
    addSelector(label, value) {
        this._selector[label] = value;
    }
    /**
     * Configure a port the service will bind to.
     *
     * This method can be called multiple times.
     *
     * @param port The port definition.
     * @stability stable
     */
    serve(port, options = {}) {
        this._ports.push({ ...options, port });
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0 && this.type !== ServiceType.EXTERNAL_NAME) {
            throw new Error('A service must be configured with a port');
        }
        if (this.type === ServiceType.EXTERNAL_NAME && this.externalName === undefined) {
            throw new Error('A service with type EXTERNAL_NAME requires an externalName prop');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                name: port.name,
                port: port.port,
                targetPort: port.targetPort ? k8s.IntOrString.fromNumber(port.targetPort) : undefined,
                nodePort: port.nodePort,
                protocol: port.protocol,
            });
        }
        return this.type !== ServiceType.EXTERNAL_NAME ? {
            clusterIp: this.clusterIP,
            externalIPs: this._externalIPs,
            externalName: this.externalName,
            type: this.type,
            selector: this._selector,
            ports: ports,
            loadBalancerSourceRanges: this._loadBalancerSourceRanges,
        } : {
            type: this.type,
            externalName: this.externalName,
        };
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "cdk8s-plus-22.Service", version: "1.0.0-beta.11" };
/**
 * @stability stable
 */
var Protocol;
(function (Protocol) {
    Protocol["TCP"] = "TCP";
    Protocol["UDP"] = "UDP";
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
//# sourceMappingURL=data:application/json;base64,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