"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IngressBackend = exports.Ingress = exports.HttpIngressPathType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
/**
 * Specify how the path is matched against request paths.
 *
 * @see https://kubernetes.io/docs/concepts/services-networking/ingress/#path-types
 * @stability stable
 */
var HttpIngressPathType;
(function (HttpIngressPathType) {
    HttpIngressPathType["PREFIX"] = "Prefix";
    HttpIngressPathType["EXACT"] = "Exact";
    HttpIngressPathType["IMPLEMENTATION_SPECIFIC"] = "ImplementationSpecific";
})(HttpIngressPathType = exports.HttpIngressPathType || (exports.HttpIngressPathType = {}));
/**
 * Ingress is a collection of rules that allow inbound connections to reach the endpoints defined by a backend.
 *
 * An Ingress can be configured to give services
 * externally-reachable urls, load balance traffic, terminate SSL, offer name
 * based virtual hosting etc.
 *
 * @stability stable
 */
class Ingress extends base_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _c;
        super(scope, id);
        this._rulesPerHost = {};
        this._tlsConfig = [];
        this.apiObject = new k8s.KubeIngress(this, 'Resource', {
            metadata: props.metadata,
            spec: {
                defaultBackend: cdk8s_1.Lazy.any({ produce: () => { var _c; return (_c = this._defaultBackend) === null || _c === void 0 ? void 0 : _c._toKube(); } }),
                rules: cdk8s_1.Lazy.any({ produce: () => this.synthRules() }),
                tls: cdk8s_1.Lazy.any({ produce: () => this.tlsConfig() }),
            },
        });
        if (props.defaultBackend) {
            this.addDefaultBackend(props.defaultBackend);
        }
        this.addRules(...(_c = props.rules) !== null && _c !== void 0 ? _c : []);
        if (props.tls) {
            this.addTls(props.tls);
        }
    }
    /**
     * (deprecated) Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        if (!this._defaultBackend && Object.keys(this._rulesPerHost).length === 0) {
            return ['ingress with no rules or default backend'];
        }
        return [];
    }
    /**
     * Defines the default backend for this ingress.
     *
     * A default backend capable of
     * servicing requests that don't match any rule.
     *
     * @param backend The backend to use for requests that do not match any rule.
     * @stability stable
     */
    addDefaultBackend(backend) {
        this.addRules({ backend });
    }
    /**
     * Specify a default backend for a specific host name.
     *
     * This backend will be used as a catch-all for requests
     * targeted to this host name (the `Host` header matches this value).
     *
     * @param host The host name to match.
     * @param backend The backend to route to.
     * @stability stable
     */
    addHostDefaultBackend(host, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend });
    }
    /**
     * Adds an ingress rule applied to requests to a specific host and a specific HTTP path (the `Host` header matches this value).
     *
     * @param host The host name.
     * @param path The HTTP path.
     * @param backend The backend to route requests to.
     * @param pathType How the path is matched against request paths.
     * @stability stable
     */
    addHostRule(host, path, backend, pathType) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend, path, pathType });
    }
    /**
     * Adds an ingress rule applied to requests sent to a specific HTTP path.
     *
     * @param path The HTTP path.
     * @param backend The backend to route requests to.
     * @param pathType How the path is matched against request paths.
     * @stability stable
     */
    addRule(path, backend, pathType) {
        this.addRules({ backend, path, pathType });
    }
    /**
     * Adds rules to this ingress.
     *
     * @param rules The rules to add.
     * @stability stable
     */
    addRules(...rules) {
        var _c, _d, _e, _f;
        for (const rule of rules) {
            // default backend is not really a rule
            if (!rule.host && !rule.path) {
                if (this._defaultBackend) {
                    throw new Error('a default backend is already defined for this ingress');
                }
                this._defaultBackend = rule.backend;
                continue;
            }
            const host = (_c = rule.host) !== null && _c !== void 0 ? _c : '';
            const backend = rule.backend;
            const path = (_d = rule.path) !== null && _d !== void 0 ? _d : '/';
            const pathType = (_e = rule.pathType) !== null && _e !== void 0 ? _e : HttpIngressPathType.PREFIX;
            if (path && !path.startsWith('/')) {
                throw new Error(`ingress paths must begin with a "/": ${path}`);
            }
            const routes = this._rulesPerHost[host] = (_f = this._rulesPerHost[host]) !== null && _f !== void 0 ? _f : [];
            // check if we already have a rule for this host/path
            if (routes.find(r => r.path === path)) {
                throw new Error(`there is already an ingress rule for ${host}${path}`);
            }
            routes.push({
                backend: backend._toKube(),
                path,
                pathType,
            });
        }
    }
    synthRules() {
        const rules = new Array();
        for (const [host, paths] of Object.entries(this._rulesPerHost)) {
            rules.push({
                host: host ? host : undefined,
                http: { paths: paths.sort(sortByPath) },
            });
        }
        return rules.length > 0 ? rules : undefined;
    }
    /**
     * @stability stable
     */
    addTls(tls) {
        this._tlsConfig.push(...tls);
    }
    tlsConfig() {
        var _c;
        if (this._tlsConfig.length == 0) {
            return undefined;
        }
        const tls = new Array();
        for (const entry of this._tlsConfig) {
            tls.push({
                hosts: entry.hosts,
                secretName: (_c = entry.secret) === null || _c === void 0 ? void 0 : _c.name,
            });
        }
        return tls;
    }
}
exports.Ingress = Ingress;
_a = JSII_RTTI_SYMBOL_1;
Ingress[_a] = { fqn: "cdk8s-plus-22.Ingress", version: "1.0.0-beta.11" };
/**
 * The backend for an ingress path.
 *
 * @stability stable
 */
class IngressBackend {
    constructor(backend) {
        this.backend = backend;
    }
    /**
     * A Kubernetes `Service` to use as the backend for this path.
     *
     * @param service The service object.
     * @stability stable
     */
    static fromService(service, options = {}) {
        if (service.ports.length === 0) {
            throw new Error('service does not expose any ports');
        }
        let servicePort;
        if (service.ports.length === 1) {
            servicePort = service.ports[0].port;
        }
        else {
            if (options.port !== undefined) {
                const found = service.ports.find(p => p.port === options.port);
                if (found) {
                    servicePort = found.port;
                }
                else {
                    throw new Error(`service exposes ports ${service.ports.map(p => p.port).join(',')} but backend is defined to use port ${options.port}`);
                }
            }
            else {
                throw new Error(`unable to determine service port since service exposes multiple ports: ${service.ports.map(x => x.port).join(',')}`);
            }
        }
        if (options.port !== undefined && servicePort !== options.port) {
            throw new Error(`backend defines port ${options.port} but service exposes port ${servicePort}`);
        }
        return new IngressBackend({
            service: {
                name: service.name,
                port: { number: servicePort },
            },
        });
    }
    /**
     * @internal
     */
    _toKube() { return this.backend; }
}
exports.IngressBackend = IngressBackend;
_b = JSII_RTTI_SYMBOL_1;
IngressBackend[_b] = { fqn: "cdk8s-plus-22.IngressBackend", version: "1.0.0-beta.11" };
function sortByPath(lhs, rhs) {
    var _c, _d;
    const p1 = (_c = lhs.path) !== null && _c !== void 0 ? _c : '';
    const p2 = (_d = rhs.path) !== null && _d !== void 0 ? _d : '';
    return p1.localeCompare(p2);
}
//# sourceMappingURL=data:application/json;base64,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