"""
HeySol API Client - Direct HTTP API operations.

This module provides a pure API client for direct HTTP interactions with HeySol services,
without MCP (Model Context Protocol) functionality.
"""

import json
from typing import Any, Dict, List, Optional

import requests

from ..config import HeySolConfig
from ..exceptions import HeySolError, ValidationError, validate_api_key_format


class HeySolAPIClient:
    """
    Pure API client for direct HTTP interactions with HeySol services.

    This client handles direct REST API calls without MCP protocol overhead.
    Use this when you need predictable, direct API access without tool discovery
    or session management complexity.
    """

    def __init__(
        self,
        api_key: Optional[str] = None,
        base_url: Optional[str] = None,
        config: Optional[HeySolConfig] = None,
    ):
        """
        Initialize the HeySol API client.

        Args:
            api_key: HeySol API key (required for authentication)
            base_url: Base URL for the API (optional, uses config default if not provided)
            config: HeySolConfig object (optional, overrides individual parameters)
        """
        # Use provided config or load from environment
        if config is None:
            config = HeySolConfig.from_env()

        # Use provided values or fall back to config
        if api_key is None:
            api_key = config.api_key
        if not base_url:
            base_url = config.base_url

        # Validate authentication
        if not api_key:
            raise ValidationError("API key is required")

        # Validate API key format and legitimacy
        validate_api_key_format(api_key)
        self._validate_api_key(api_key, base_url)

        self.api_key = api_key
        self.base_url = base_url
        self.source = config.source
        self.profile_url = config.profile_url
        self.timeout = config.timeout

    @classmethod
    def from_env(cls) -> "HeySolAPIClient":
        """
        Create API client from environment variables.

        Returns:
            HeySolAPIClient: Configured API client instance
        """
        config = HeySolConfig.from_env()
        return cls(config=config)

    def _validate_api_key(self, api_key: str, base_url: str) -> None:
        """
        Validate API key by making a test API call.

        Args:
            api_key: The API key to validate
            base_url: The base URL for API calls

        Raises:
            ValidationError: If the API key is invalid or authentication fails
        """

        try:
            # Make a test request to get spaces (lightweight endpoint)
            test_url = base_url.rstrip("/") + "/spaces"
            headers = {
                "Content-Type": "application/json",
                "Accept": "application/json",
                "Authorization": f"Bearer {api_key}",
            }

            response = requests.get(
                url=test_url,
                headers=headers,
                timeout=10,  # Short timeout for validation
            )

            # Check for authentication errors
            if response.status_code == 401:
                raise ValidationError("Invalid API key or authentication failed")
            elif response.status_code == 403:
                raise ValidationError("API key does not have required permissions")
            elif response.status_code >= 400:
                # For other client errors, assume key is invalid
                raise ValidationError(f"API key validation failed: HTTP {response.status_code}")

            # If we get here, the key is valid (even if spaces endpoint returns data or not)
            response.raise_for_status()

        except requests.exceptions.RequestException as e:
            if "401" in str(e) or "403" in str(e):
                raise ValidationError("Invalid API key or authentication failed") from e
            else:
                # For network/other errors, we'll allow the key for now
                # The actual API calls will fail later if the key is truly invalid
                pass

    def _get_authorization_header(self) -> str:
        """Get the authorization header using API key."""
        if not self.api_key:
            raise HeySolError("No API key available for authentication")
        return f"Bearer {self.api_key}"

    def _is_valid_id_format(self, id_value: str) -> bool:
        """Validate if an ID has a reasonable format (not obviously invalid)."""
        if not id_value or len(id_value.strip()) == 0:
            return False

        # Check for obviously invalid patterns
        invalid_patterns = [
            "invalid",
            "test",
            "null",
            "none",
            "undefined",
            " ",
            "\t",
            "\n",  # whitespace-only
        ]

        id_lower = id_value.lower().strip()
        for pattern in invalid_patterns:
            if pattern in id_lower:
                return False

        # Should contain some alphanumeric characters and not be too short
        if len(id_value) < 3:
            return False

        return True

    def _make_request(
        self,
        method: str,
        endpoint: str,
        data: Optional[Dict[str, Any]] = None,
        params: Optional[Dict[str, Any]] = None,
    ) -> Dict[str, Any]:
        """Make a direct HTTP request to the API."""
        # Handle absolute URLs (for endpoints that need different base URLs)
        if endpoint.startswith("http"):
            url = endpoint
        else:
            url = self.base_url.rstrip("/") + "/" + endpoint.lstrip("/")

        headers = {
            "Content-Type": "application/json",
            "Accept": "application/json",
            "Authorization": self._get_authorization_header(),
        }

        response = requests.request(
            method=method,
            url=url,
            json=data,
            params=params,
            headers=headers,
            timeout=self.timeout,
        )

        response.raise_for_status()
        return response.json()

    def ingest(
        self,
        message: str,
        space_id: Optional[str] = None,
        session_id: Optional[str] = None,
        source: Optional[str] = None,
    ) -> Dict[str, Any]:
        """Ingest data into CORE Memory using direct API."""
        if not message:
            raise ValidationError("Message is required for ingestion")

        # Use direct API call with the correct format
        payload: Dict[str, Any] = {
            "episodeBody": message,
            "referenceTime": "2023-11-07T05:31:56Z",  # Use current timestamp
            "metadata": {},
            "source": source or self.source or "heysol-api-client",
            "sessionId": session_id or "",
        }
        if space_id:
            payload["spaceId"] = space_id

        return self._make_request("POST", "add", data=payload)

    def copy_log_entry(
        self,
        log_entry: Dict[str, Any],
        new_source: Optional[str] = None,
        new_space_id: Optional[str] = None,
        new_session_id: Optional[str] = None,
        override_metadata: Optional[Dict[str, Any]] = None,
    ) -> Dict[str, Any]:
        """Copy a log entry with all metadata preserved, allowing selective field overrides.

        Args:
            log_entry: The original log entry to copy
            new_source: Optional new source identifier (preserves original if None)
            new_space_id: Optional new space ID (preserves original if None)
            new_session_id: Optional new session ID (preserves original if None)
            override_metadata: Optional metadata fields to override (merges with original)

        Returns:
            The response from the copy operation
        """
        if not log_entry:
            raise ValidationError("Log entry is required for copying")

        # Extract message content
        message_content = (
            log_entry.get("ingestText")
            or log_entry.get("data", {}).get("episodeBody")
            or log_entry.get("episodeBody", "")
        )
        if not message_content:
            raise ValidationError("Log entry must contain message content")

        # Extract original metadata and timestamp to preserve them
        original_data = log_entry.get("data", {})
        original_metadata = original_data.get("metadata", {})
        original_reference_time = (
            original_data.get("referenceTime") or log_entry.get("time") or "2023-11-07T05:31:56Z"
        )
        original_session_id = original_data.get("sessionId") or log_entry.get("sessionId") or ""

        # Merge metadata if overrides provided
        final_metadata = original_metadata.copy()
        if override_metadata:
            final_metadata.update(override_metadata)

        # Build payload with preserved fields
        payload = {
            "episodeBody": message_content,
            "referenceTime": original_reference_time,
            "metadata": final_metadata,
            "source": new_source or log_entry.get("source", "heysol-api-client"),
            "sessionId": new_session_id or original_session_id,
        }

        # Add space ID if specified
        space_id = new_space_id
        if space_id:
            payload["spaceId"] = space_id

        return self._make_request("POST", "add", data=payload)

    def search(
        self,
        query: str,
        space_ids: Optional[List[str]] = None,
        limit: int = 10,
        include_invalidated: bool = False,
    ) -> Dict[str, Any]:
        """Search for memories in CORE Memory using direct API."""
        if not query:
            raise ValidationError("Search query is required")

        # Use direct API call with correct format
        payload = {
            "query": query,
            "spaceIds": space_ids or [],
            "includeInvalidated": include_invalidated,
        }

        params = {"limit": limit}

        return self._make_request("POST", "search", data=payload, params=params)

    def get_spaces(self) -> List[Any]:
        """Get available memory spaces using direct API."""
        result = self._make_request("GET", "spaces")
        return result.get("spaces", result) if isinstance(result, dict) else result

    def create_space(self, name: str, description: str = "") -> Optional[str]:
        """Create a new memory space."""
        if not name:
            raise ValidationError("Space name is required")

        payload = {"name": name, "description": description}
        data = self._make_request("POST", "spaces", data=payload)

        # Handle different response formats
        space_id = None
        if isinstance(data, dict):
            space_id = data.get("space", {}).get("id") or data.get("id") or data.get("space_id")
        return space_id

    def get_user_profile(self) -> Dict[str, Any]:
        """Get the current user's profile.

        Note: This endpoint requires OAuth authentication which is not yet available on the server side.
        This method will fail with a 404 error until OAuth is implemented.
        Currently, only API key authentication is supported for other endpoints.
        """
        # Profile endpoint uses a different URL than the base API URL
        return self._make_request("GET", self.profile_url)

    # Memory endpoints
    def search_knowledge_graph(
        self, query: str, space_id: Optional[str] = None, limit: int = 10, depth: int = 2
    ) -> Dict[str, Any]:
        """Search the knowledge graph for related concepts and entities."""
        if not query:
            raise ValidationError("Search query is required")

        if limit < 1 or limit > 100:
            raise ValidationError("Limit must be between 1 and 100")

        if depth < 1 or depth > 5:
            raise ValidationError("Depth must be between 1 and 5")

        # Use the same search endpoint but with knowledge graph parameters
        payload = {
            "query": query,
            "spaceIds": [space_id] if space_id else [],
            "includeInvalidated": False,
        }

        params = {"limit": limit, "depth": depth, "type": "knowledge_graph"}

        result = self._make_request("POST", "search", data=payload, params=params)
        return result

    def add_data_to_ingestion_queue(
        self,
        data: Any,
        space_id: Optional[str] = None,
        priority: str = "normal",
        tags: Optional[List[str]] = None,
        metadata: Optional[Dict[str, Any]] = None,
    ) -> Dict[str, Any]:
        """Add data to the ingestion queue for processing."""
        # Handle different data formats - extract content as episodeBody
        if isinstance(data, str):
            episode_body = data
        elif isinstance(data, dict):
            episode_body = data.get("content", json.dumps(data))
        else:
            episode_body = str(data)

        # Use the same /add endpoint as ingest with minimal payload
        payload = {
            "episodeBody": episode_body,
            "referenceTime": "2023-11-07T05:31:56Z",
            "metadata": metadata or {},
            "source": self.source or "heysol-api-client",
            "sessionId": "",
        }
        if space_id:
            payload["spaceId"] = space_id

        result = self._make_request("POST", "add", data=payload)
        return result

    def get_episode_facts(
        self, episode_id: str, limit: int = 100, offset: int = 0, include_metadata: bool = True
    ) -> Dict[str, Any]:
        """Get episode facts from CORE Memory."""
        if not episode_id:
            raise ValidationError("Episode ID is required")

        params = {"limit": limit, "offset": offset, "include_metadata": include_metadata}
        return self._make_request("GET", f"episodes/{episode_id}/facts", params=params)

    def get_ingestion_logs(
        self,
        space_id: Optional[str] = None,
        limit: int = 100,
        offset: int = 0,
        status: Optional[str] = None,
        start_date: Optional[str] = None,
        end_date: Optional[str] = None,
    ) -> List[Any]:
        """Get ingestion logs from CORE Memory."""
        params: Dict[str, Any] = {"limit": limit, "offset": offset}
        if space_id:
            params["spaceId"] = space_id
        if status:
            params["status"] = status
        if start_date:
            params["startDate"] = start_date
        if end_date:
            params["endDate"] = end_date

        try:
            result = self._make_request("GET", "logs", params=params)
            return result.get("logs", result) if isinstance(result, dict) else result
        except Exception as e:
            # If logs endpoint fails, return empty list with a note
            print(f"Warning: Logs endpoint not available: {e}")
            return []

    def iter_ingestion_logs(
        self,
        space_id: Optional[str] = None,
        status: Optional[str] = None,
        start_date: Optional[str] = None,
        end_date: Optional[str] = None,
    ):
        """
        Generator that yields all ingestion logs for memory-efficient processing.

        Args:
            space_id: Optional space ID to filter logs
            status: Optional status filter
            start_date: Optional start date filter
            end_date: Optional end date filter

        Yields:
            Individual log entries as they are fetched
        """
        offset = 0
        batch_size = 1000  # Backend batch size for efficient API calls

        while True:
            batch = self.get_ingestion_logs(
                space_id=space_id,
                limit=batch_size,
                offset=offset,
                status=status,
                start_date=start_date,
                end_date=end_date,
            )

            if not batch:
                break

            for log in batch:
                yield log

            # If we got fewer logs than requested, we've reached the end
            if len(batch) < batch_size:
                break

            offset += batch_size

            # Safety check to prevent infinite loops (max 1000 batches = 1,000,000 logs)
            if offset >= 1000000:
                break

    def get_logs_by_source(
        self,
        source: str,
        space_id: Optional[str] = None,
        limit: int = 100,
        offset: int = 0,
        status: Optional[str] = None,
        start_date: Optional[str] = None,
        end_date: Optional[str] = None,
    ) -> List[Any]:
        """Get ingestion logs filtered by source using efficient streaming."""
        if not source:
            raise ValidationError("Source is required for filtering logs")

        # Use the generator for memory-efficient processing
        filtered_logs = []
        collected_count = 0

        for log in self.iter_ingestion_logs(
            space_id=space_id,
            status=status,
            start_date=start_date,
            end_date=end_date,
        ):
            # Filter by source
            if log.get("source") == source:
                collected_count += 1

                # Skip logs before offset
                if collected_count <= offset:
                    continue

                # Add to results
                filtered_logs.append(log)

                # Stop if we've reached the limit
                if limit > 0 and len(filtered_logs) >= limit:
                    break

        return filtered_logs

    def get_specific_log(self, log_id: str) -> Dict[str, Any]:
        """Get a specific ingestion log by ID."""
        if not log_id:
            raise ValidationError("Log ID is required")

        # Validate log ID format
        if not self._is_valid_id_format(log_id):
            raise ValidationError(f"Invalid log ID format: {log_id}")

        try:
            result = self._make_request("GET", f"logs/{log_id}")
            return result.get("log", result) if isinstance(result, dict) else result
        except Exception as e:
            # If specific log endpoint fails, return error info
            return {
                "error": f"Log retrieval failed: {e}",
                "log_id": log_id,
                "note": "Log status checking may not be available via current API",
            }

    def check_ingestion_status(
        self, run_id: Optional[str] = None, space_id: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Check the status of data ingestion processing.

        Since the logs endpoint may not be available, this method provides
        alternative ways to check processing status.

        Args:
            run_id: Run ID from ingestion response (if available)
            space_id: Space ID to check for recent activity

        Returns:
            Dict with status information and recommendations
        """
        status_info: Dict[str, Any] = {
            "ingestion_status": "unknown",
            "recommendations": [],
            "available_methods": [],
        }

        if space_id:
            status_info["space_id"] = space_id

        # Try to get logs if endpoint is available
        try:
            logs = self.get_ingestion_logs(space_id=space_id, limit=5)
            if logs and len(logs) > 0:
                status_info["ingestion_status"] = "logs_available"
                status_info["recent_logs_count"] = len(logs)
                status_info["available_methods"].append("get_ingestion_logs")
            else:
                status_info["ingestion_status"] = "no_logs_found"
        except Exception as e:
            status_info["logs_error"] = str(e)

        # Try search to see if data is available
        try:
            search_result = self.search("test", space_ids=[space_id] if space_id else None, limit=1)
            episodes = search_result.get("episodes", [])
            if episodes:
                status_info["search_status"] = "data_available"
                status_info["available_methods"].append("search")
            else:
                status_info["search_status"] = "no_search_results"
        except Exception as e:
            status_info["search_error"] = str(e)

        # Provide recommendations based on what works
        if "logs_available" in status_info.get("ingestion_status", ""):
            status_info["recommendations"].append(
                "Use get_ingestion_logs() to check processing status"
            )
        elif "data_available" in status_info.get("search_status", ""):
            status_info["recommendations"].append(
                "Data appears to be processed - use search() to verify"
            )
        else:
            status_info["recommendations"].extend(
                [
                    "Wait a few minutes for data processing to complete",
                    "Use search() with your ingested content to check if it's available",
                    "Check the HeySol dashboard for processing status",
                ]
            )

        return status_info

    # Spaces endpoints
    def bulk_space_operations(
        self,
        intent: str,
        space_id: Optional[str] = None,
        statement_ids: Optional[List[str]] = None,
        space_ids: Optional[List[str]] = None,
    ) -> Dict[str, Any]:
        """Perform bulk operations on spaces."""
        if not intent:
            raise ValidationError("Intent is required for bulk operations")

        payload: Dict[str, Any] = {"intent": intent}
        if space_id:
            payload["spaceId"] = space_id
        if statement_ids:
            payload["statementIds"] = statement_ids
        if space_ids:
            payload["spaceIds"] = space_ids

        return self._make_request("PUT", "spaces", data=payload)

    def get_space_details(
        self, space_id: str, include_stats: bool = True, include_metadata: bool = True
    ) -> Dict[str, Any]:
        """Get detailed information about a specific space."""
        if not space_id:
            raise ValidationError("Space ID is required")

        # Validate space ID format (should be a valid identifier)
        if not self._is_valid_id_format(space_id):
            raise ValidationError(f"Invalid space ID format: {space_id}")

        params = {"include_stats": include_stats, "include_metadata": include_metadata}
        return self._make_request("GET", f"spaces/{space_id}", params=params)

    def update_space(
        self,
        space_id: str,
        name: Optional[str] = None,
        description: Optional[str] = None,
        metadata: Optional[Dict[str, Any]] = None,
    ) -> Dict[str, Any]:
        """Update properties of an existing space."""
        if not space_id:
            raise ValidationError("Space ID is required")

        payload = {}
        if name is not None:
            payload["name"] = name
        if description is not None:
            payload["description"] = description
        # Note: API documentation shows only name and description are supported
        # metadata parameter is kept for future compatibility but not sent

        if not payload:
            raise ValidationError("At least one field must be provided for update")

        return self._make_request("PUT", f"spaces/{space_id}", data=payload)

    def delete_space(self, space_id: str, confirm: bool = False) -> Dict[str, Any]:
        """Delete a space."""
        if not space_id:
            raise ValidationError("Space ID is required")

        if not confirm:
            raise ValidationError("Space deletion requires confirmation (confirm=True)")

        return self._make_request("DELETE", f"spaces/{space_id}")

    # Webhook endpoints
    def register_webhook(
        self, url: str, events: Optional[List[str]] = None, secret: str = ""
    ) -> Dict[str, Any]:
        """Register a new webhook."""
        if not url:
            raise ValidationError("Webhook URL is required")

        if secret == "":
            raise ValidationError("Webhook secret is required")

        # Use form data format as specified in API docs (only url and secret)
        data = {"url": url, "secret": secret}

        # Create a custom request for form data
        request_url = self.base_url.rstrip("/") + "/" + "webhooks".lstrip("/")

        headers = {
            "Authorization": self._get_authorization_header(),
            "Content-Type": "application/x-www-form-urlencoded",
            "Accept": "application/json",
        }

        response = requests.post(
            url=request_url,
            data=data,  # This will automatically encode as form data
            headers=headers,
            timeout=self.timeout,
        )

        response.raise_for_status()
        return response.json()

    def list_webhooks(
        self,
        space_id: Optional[str] = None,
        active: Optional[bool] = None,
        limit: int = 100,
        offset: int = 0,
    ) -> List[Any]:
        """List all webhooks."""
        params = {"limit": limit, "offset": offset}
        # Note: space_id and active parameters may not be supported by the API
        # if space_id:
        #     params["space_id"] = space_id
        # if active is not None:
        #     params["active"] = active

        try:
            result = self._make_request("GET", "webhooks", params=params)
            return result.get("webhooks", result) if isinstance(result, dict) else result
        except Exception as e:
            # If webhooks endpoint fails, return empty list with a note
            print(f"Warning: Webhooks endpoint not available: {e}")
            return []

    def get_webhook(self, webhook_id: str) -> Dict[str, Any]:
        """Get webhook details."""
        if not webhook_id:
            raise ValidationError("Webhook ID is required")

        # Validate webhook ID format
        if not self._is_valid_id_format(webhook_id):
            raise ValidationError(f"Invalid webhook ID format: {webhook_id}")

        return self._make_request("GET", f"webhooks/{webhook_id}")

    def update_webhook(
        self, webhook_id: str, url: str, events: List[str], secret: str = "", active: bool = True
    ) -> Dict[str, Any]:
        """Update webhook properties."""
        if not webhook_id:
            raise ValidationError("Webhook ID is required")

        if not url:
            raise ValidationError("Webhook URL is required")

        if not events:
            raise ValidationError("Webhook events are required")

        if not secret:
            raise ValidationError("Webhook secret is required")

        # Use form data format as specified in API
        data = {
            "url": url,
            "events": ",".join(events),
            "secret": secret,
            "active": str(active).lower(),
        }

        # Create a custom request for form data
        request_url = self.base_url.rstrip("/") + "/" + f"webhooks/{webhook_id}".lstrip("/")

        headers = {
            "Authorization": self._get_authorization_header(),
            "Content-Type": "application/x-www-form-urlencoded",
            "Accept": "application/json",
        }

        response = requests.put(
            url=request_url,
            data=data,  # This will automatically encode as form data
            headers=headers,
            timeout=self.timeout,
        )

        response.raise_for_status()
        return response.json()

    def delete_webhook(self, webhook_id: str, confirm: bool = False) -> Dict[str, Any]:
        """Delete a webhook."""
        if not webhook_id:
            raise ValidationError("Webhook ID is required")

        if not confirm:
            raise ValidationError("Webhook deletion requires confirmation (confirm=True)")

        return self._make_request("DELETE", f"webhooks/{webhook_id}")

    def delete_log_entry(self, log_id: str) -> Dict[str, Any]:
        """Delete a log entry from CORE Memory."""
        if not log_id:
            raise ValidationError("Log ID is required")

        # Use the DELETE endpoint with log ID in payload
        payload = {"id": log_id}
        return self._make_request("DELETE", f"logs/{log_id}", data=payload)

    def close(self) -> None:
        """Close the API client and clean up resources."""
        # Currently no resources to clean up, but method provided for API compatibility
        pass
